package ru.yandex.tours.util.spray

import org.joda.time.LocalDate
import org.joda.time.format.DateTimeFormat
import shapeless.{::, HNil}
import spray.routing.Directives._
import spray.routing._

import scala.util.{Failure, Success, Try}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.02.15
 */
trait CommonDirectives {

  private val format = "yyyy-MM-dd"
  private val formatter = DateTimeFormat.forPattern(format)

  def range[T](name: String, parse: String => T): Directive[Option[T] :: Option[T] :: HNil] = {
    parameters(s"${name}_from".?, s"${name}_to".?) hflatMap {
      case stringFrom :: stringTo :: HNil =>
        Try(stringFrom.map(parse)) match {
          case Success(from) =>
            Try(stringTo.map(parse)) match {
              case Success(to) =>
                hprovide(from :: to :: HNil)
              case Failure(e) => reject(MalformedQueryParamRejection(name + "_to", "Can not parse"))
            }
          case Failure(e) => reject(MalformedQueryParamRejection(name + "_from", "Can not parse"))
        }
    }
  }

  def date(name: String): Directive1[LocalDate] = optDate(name) flatMap {
    case Some(date) => provide(date)
    case None => reject(MissingQueryParamRejection(name))
  }

  def optDate(name: String): Directive1[Option[LocalDate]] = parameter(name.?) flatMap {
    case Some(date) =>
      Try(LocalDate.parse(date, formatter)) match {
        case Success(parsedDate) =>
          provide(Some(parsedDate))
        case _ => reject(MalformedQueryParamRejection(name, s"$name should be in $format format"))
      }
    case None => provide(None)
  }

  def array(name: String, isEmptyOk: Boolean = false): Directive1[Seq[String]] = parameterMultiMap flatMap {
    map =>
      val array = map.getOrElse(name, Seq.empty[String]).flatMap(_.split(","))
      if (array.nonEmpty || isEmptyOk) {
        provide(array)
      } else {
        reject(MalformedQueryParamRejection(name, s"At least one value should present"))
      }
  }

  def intArray(name: String, isEmptyOk: Boolean = false): Directive1[Seq[Int]] = array(name, isEmptyOk) flatMap {
    values =>
      val array = values.map(t => Try(t.toInt))
      array.find(_.isFailure) match {
        case Some(failed) => reject(MalformedQueryParamRejection(name, s"parameter expected to be int"))
        case None => provide(array.map(_.get))
      }
  }

  def longArray(name: String, isEmptyOk: Boolean = false): Directive1[Seq[Long]] = array(name, isEmptyOk) flatMap {
    values =>
      val array = values.map(t => Try(t.toLong))
      array.find(_.isFailure) match {
        case Some(failed) => reject(MalformedQueryParamRejection(name, s"parameter expected to be long"))
        case None => provide(array.map(_.get))
      }
  }

  def enum[E](name: String, values: Iterable[E], default: Option[E] = None): Directive1[E] = {
    val map = values.map(v => v.toString.toUpperCase -> v).toMap
    parameter(name.?) flatMap {
      case Some(param) =>
        map.get(param.toUpperCase) match {
          case Some(value) => provide(value)
          case None =>
            val known = map.keys.mkString(", ")
            reject(MalformedQueryParamRejection(name, s"Unknown $name parameter: $param. Known values are: $known"))
        }
      case None =>
        default match {
          case Some(value) => provide(value)
          case None => reject(MissingQueryParamRejection(name))
        }
    }
  }

  def enumArray[E](name: String, values: Array[E], isEmptyOk: Boolean = false): Directive1[Seq[E]] = {
    val map = values.map(v => v.toString.toUpperCase -> v).toMap
    array(name, isEmptyOk) flatMap { optValues =>
      val invalid = Seq.newBuilder[String]
      val parsed = Seq.newBuilder[E]
      optValues.foreach { raw =>
        map.get(raw.toUpperCase) match {
          case Some(value) => parsed += value
          case None => invalid += raw
        }
      }
      if (invalid.result().isEmpty) provide(parsed.result())
      else {
        val unknown = invalid.result().mkString(", ")
        val known = map.keySet.mkString(", ")
        reject(MalformedQueryParamRejection(name, s"Unknown $name parameter: $unknown. Known values are: $known"))
      }
    }
  }
}

object CommonDirectives extends CommonDirectives