package ru.yandex.tours.util.spray

import ru.yandex.tours.metro.{MetroHolder, MetroUtil}
import ru.yandex.tours.model.Agencies.Metro
import ru.yandex.tours.model.util.SortType
import ru.yandex.tours.model.util.SortType._
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.model.{Languages, MapRectangle}
import ru.yandex.tours.model.Languages._
import shapeless.{::, HNil}
import spray.routing.Directives._
import spray.routing._

import scala.util.{Failure, Success, Try}

trait CommonModelDirectives extends SearchDirectives with CommonDirectives {

  val sortBy: Directive1[SortType] = enum("sort", SortType.values, default = Some(SortType.RELEVANCE))

  val lang: Directive1[Lang] = enum("lang", Languages.values, default = Some(Languages.ru))

  val spanMapRectangle: Directive1[MapRectangle] = {
    parameters('longitude.as[Double], 'latitude.as[Double], 'lon_span.as[Double], 'lat_span.as[Double]) hflatMap {
      case lon :: lat :: lonSpan :: latSpan :: HNil =>
        Try(MapRectangle.byCenterAndSpan(lon, lat, lonSpan, latSpan)) match {
          case Success(mr) => provide(mr)
          case Failure(e) => reject(MalformedQueryParamRejection("map rectangle", e.getMessage))
        }
    }
  }

  val boundingMapRectangle: Directive1[MapRectangle] = {
    parameters('lat_min.as[Double], 'lat_max.as[Double], 'lon_min.as[Double], 'lon_max.as[Double]) hflatMap {
      case latMin :: latMax :: lonMin :: lonMax :: HNil =>
        Try(MapRectangle.byBoundaries(lonMin, latMin, lonMax, latMax)) match {
          case Success(mr) => provide(mr)
          case Failure(e) => reject(MalformedQueryParamRejection("map rectangle", e.getMessage))
        }
    }
  }

  val mapRectangle: Directive1[MapRectangle] = spanMapRectangle | boundingMapRectangle

  def optMetro(metroHolder: MetroHolder): Directive1[Option[Metro]] = parameter('metro.?) flatMap {
    case Some(metro) ⇒
      val parts = metro.split(MetroUtil.delimiter)
      Try(parts(0).toInt) match {
        case Success(metroId) =>
          var candidates = metroHolder.byId(metroId)
          if (parts.size == 2) {
            val color = parts(1)
            candidates = candidates.filter(_.getColor == color)
          }
          if (candidates.size == 1) {
            provide(Some(candidates.head))
          } else {
            reject(MalformedQueryParamRejection("metro", s"Unknown metro with id $metro"))
          }
        case Failure(e) => reject(MalformedQueryParamRejection("metro", "metro param should start with Int"))
      }
    case None => provide(None)
  }


  val utm: Directive1[UtmMark] = {
    parameters('utm_source.?, 'utm_medium.?, 'utm_campaign.?, 'utm_content.?, 'utm_term.?, 'user_query.?, 'req_id.?) hmap {
      case source :: medium :: campaign :: content :: term :: userQuery :: reqId :: HNil =>
        UtmMark(source, medium, campaign, content, term, userQuery, reqId)
    }
  }
}

object CommonModelDirectives extends CommonModelDirectives