package ru.yandex.tours.util.spray

import org.json.JSONArray
import ru.yandex.common.actor.InstrumentedActor
import ru.yandex.tours.util.http.HttpTimeout
import spray.http.{IllegalRequestException, RequestProcessingException, StatusCodes}
import spray.routing._

import scala.util.control.NonFatal

abstract class HttpHandler extends HttpServiceActor with InstrumentedActor {
  implicit final val ac = context.dispatcher
  implicit val rs: RoutingSettings = RoutingSettings.default

  def route: Route

  implicit val rejectionHander = new RejectionHandler {
    override def isDefinedAt(x: List[Rejection]): Boolean = true

    override def apply(v1: List[Rejection]): Route = {
      val array = new JSONArray()
      if (v1.nonEmpty) {
        v1.foreach(array.put)
      } else {
        array.put("The requested resource could not be found.")
      }
      completeJsonError(StatusCodes.BadRequest, array)
    }
  }

  override protected def process: PartialFunction[Any, Unit] = runRoute(route)

  override protected def healthyEventPeriod = Option.empty

  private implicit val exceptionHandler = ExceptionHandler.apply {
    case e: IllegalRequestException => ctx =>
      slf4jLog.warn(s"Illegal request ${ctx.request}. ${e.getMessage}. Completing with '${e.status}' response")
      completeJsonError(e.status, e.info.format(rs.verboseErrorMessages))(ctx)

    case e: RequestProcessingException => ctx =>
      slf4jLog.warn(s"Request ${ctx.request} could not be handled normally. Completing with '${e.status}' response", e)
      completeJsonError(e.status, e.info.format(rs.verboseErrorMessages))(ctx)

    case NonFatal(e) => ctx =>
      e match {
        case HttpTimeout(_) =>
          log.error(s"Timeout during processing of request ${ctx.request}: ${e.getMessage}")
          completeJsonError(StatusCodes.GatewayTimeout, s"${e.getMessage}! See logs")(ctx)
        case _ =>
          log.error(e, s"Error during processing of request ${ctx.request}")
          completeJsonError(StatusCodes.InternalServerError, s"${e.getMessage}! See logs")(ctx)
      }
  }


}
