package ru.yandex.tours.util.spray

import java.util.concurrent.ConcurrentHashMap

import akka.actor.{Cancellable, ActorSystem}
import akka.util.Timeout
import com.typesafe.config.Config
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.http.AsyncHttpClient
import spray.http.{Uri, StatusCodes}
import spray.routing.Directives._

import scala.collection.JavaConverters._
import scala.concurrent.duration._
import scala.util.Success

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 27.05.15
 */
class PingControl extends Logging {

  private var disabledPings = false
  private val remoteCheckFailed = new ConcurrentHashMap[String, Boolean]()

  def scheduleRemoteCheck(config: Config,
                          httpClient: AsyncHttpClient,
                          interval: FiniteDuration,
                          timeout: FiniteDuration)(implicit akkaSystem: ActorSystem): Any = {
    val host = config.getString("host")
    val port = config.getString("port")
    val url = s"http://$host:$port/ping"

    scheduleRemoteCheck(url, httpClient, interval, timeout)
  }

  def scheduleRemoteCheck(url: String,
                          httpClient: AsyncHttpClient,
                          interval: FiniteDuration,
                          timeout: FiniteDuration)(implicit akkaSystem: ActorSystem): Cancellable = {
    import akkaSystem.dispatcher
    val tout = Timeout(timeout)
    val uri = Uri(url)
    akkaSystem.scheduler.schedule(interval, interval) {
      httpClient.get(uri)(tout).onComplete {
        case Success((code, _)) if code.isSuccess =>
          val lastCheck = remoteCheckFailed.getOrDefault(url, false)
          if (lastCheck) log.info(s"Enabled /ping – got result from [$uri]")
          remoteCheckFailed.put(url, false)
        case res =>
          val lastCheck = remoteCheckFailed.getOrDefault(url, false)
          if (!lastCheck) log.info(s"Disabled /ping – failed to check [$uri]: $res")
          remoteCheckFailed.put(url, true)
      }
    }
  }

  lazy val route =
    pathEndOrSingleSlash {
      dynamic {
        if (disabledPings || remoteCheckFailed.asScala.exists(_._2)) {
          complete(StatusCodes.ServiceUnavailable, "2;Service Unavailable")
        } else {
          complete(StatusCodes.OK, "0;OK")
        }
      }
    } ~ host("localhost") {
      path("enable") {
        dynamic {
          log.info("Enabled /ping handler")
          disabledPings = false
          complete(StatusCodes.OK, "Enabled /ping")
        }
      } ~ path("disable") {
        dynamic {
          log.info("Disabled /ping handler")
          disabledPings = true
          complete(StatusCodes.OK, "Disabled /ping")
        }
      }
    }
}
