package ru.yandex.tours.util.spray

import java.util.UUID

import org.joda.time.LocalDate
import ru.yandex.tours.calendar.Calendar.FlightDay
import ru.yandex.tours.geo.Departures
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.search._
import ru.yandex.tours.model.util.DateInterval
import ru.yandex.tours.search.{DefaultRequestGenerator, Defaults}
import ru.yandex.tours.services.CalendarService
import ru.yandex.tours.util.lang.Dates._
import ru.yandex.tours.util.spray.CommonDirectives._
import shapeless.{::, HNil}
import spray.http.StatusCodes
import spray.routing.Directives._
import spray.routing.{Directive1, MalformedQueryParamRejection, MissingQueryParamRejection}

import scala.concurrent.{ExecutionContext, Future}
import scala.util.Success

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 24.02.16
 */
class RouteesContext(departures: Departures,
                     tree: region.Tree,
                     calendarService: Option[CalendarService])(implicit ec: ExecutionContext) {

  private val MaxAgesSize = 20

  private val baseDirective = {
    optDate("when") &
      parameter('from.as[Int].?) &
      parameter('to.as[Int].?) &
      parameter('nights.as[Int].?) &
      intArray("ages", isEmptyOk = true) &
      userRegion &
      parameters('when_flex.as[Boolean] ? false) &
      parameter('nights_flex.as[Boolean] ? false) &
      enum("currency", Currency.values, default = Some(Currency.RUB)) &
      enum[Languages.Lang]("lang", Languages.values, default = Some(Languages.ru)) &
      parameter('search_filter.as[String] ? "") &
      parameter('unique_id.as[String] ? "") &
      parameters('utm_source.?, 'utm_medium.?, 'utm_campaign.?, 'utm_content.?, 'utm_term.?)
  }

  val searchRequest: Directive1[HotelSearchRequest] = searchRequest(None).flatMap {
    case req if tree.region(req.from).isEmpty && req.from > 0 =>
      complete(StatusCodes.NotFound, s"Unknown region `from`: ${req.from}")
    case req if tree.region(req.to).isEmpty && req.to > 0 =>
      complete(StatusCodes.NotFound, s"Unknown region `to`: ${req.to}")
    case req => provide(req)
  }

  def searchRequest(defaultTo: Int): Directive1[HotelSearchRequest] = searchRequest(Some(defaultTo))

  def searchRequestWithDestination(to: Int): Directive1[HotelSearchRequest] = searchRequest(Some(to)).map {
    _.copy(to = to)
  }

  // not sure about it
  val roomSearchRequest: Directive1[HotelSearchRequest] = searchRequest.map {
    _.copy(from = 0, flexWhen = false, flexNights = false)
  }

  val flightSearchRequest: Directive1[FlightSearchRequest] = (searchRequest & parameter('airport_id)) hmap {
    case request :: airportId :: HNil => FlightSearchRequest(request, airportId)
  }

  val transferSearchRequest: Directive1[TransferSearchRequest] = {
    (searchRequest & parameters('airport_id, 'hotel_id.as[Int]) & SearchDirectives.userIp).hmap {
      case request :: airportId :: hotelId :: userIp :: HNil =>
        TransferSearchRequest(
          request.copy(from = 0, flexWhen = false, flexNights = false),
          hotelId, airportId, userIp)
    }
  }

  def offersSearchRequest(hotel: Hotel): Directive1[OfferSearchRequest] =
    searchRequest(defaultTo = hotel.geoId)
      .map { req => if (tree.pathToRoot(hotel.geoId).exists(_.id == req.to)) req else req.copy(to = hotel.geoId) }
      .map { req => OfferSearchRequest(req, hotel.id) }


  private def searchRequest(defaultTo: Option[Int] = None): Directive1[HotelSearchRequest] = {
    baseDirective.hflatMap {
      case optWhen :: optFrom :: optTo :: optNights :: optAges :: optGeoId :: whenFlex :: nightsFlex ::
        currency :: lang :: filterStr :: uniqueId :: source :: medium :: campaign :: content :: term :: HNil =>

        val optToPositive = optTo.filter(_ >= 0)
        if (optToPositive.isEmpty && defaultTo.isEmpty) {
          reject(MissingQueryParamRejection("to"))
        } else if (optAges.exists(_ <= 0)) {
          reject(MalformedQueryParamRejection("ages", "ages must be positive"))
        } else if (optAges.size > MaxAgesSize) {
          reject(MalformedQueryParamRejection("ages", "too many ages"))
        } else {
          val to = optToPositive.orElse(defaultTo).get
          val from = resolveFrom(to, optFrom, optGeoId)
          val ages = if (optAges.isEmpty) Defaults.TWO_ADULTS else optAges
          val filter = SearchFilter.parse(filterStr)
          val uniqueIdChecked = if (uniqueId.equals("")) UUID.randomUUID().toString else uniqueId
          resolveFlightDay(from, to, optNights, optWhen).map { fd =>
            HotelSearchRequest(from, to, fd.nights, fd.when, ages, whenFlex, nightsFlex, currency, lang, filter,
              uniqueIdChecked, source, medium, campaign, content, term)
          }
        }
    }
  }

  private def userRegion = SearchDirectives.userRegion

  private def resolveFrom(to: Int, optFrom: Option[Int], optGeoId: Option[Int]): Int = {
    optFrom.getOrElse {
      optGeoId.fold(Defaults.MOSCOW_GEO_ID)(departures.getBestDeparture(_, to))
    }

    optFrom match {
      case Some(from) => from
      case None =>
        optGeoId match {
          case Some(userGeoId) => departures.getBestDeparture(userGeoId, to)
          case _ => DefaultRequestGenerator.MOSCOW_GEO_ID
        }
    }
  }

  private def resolveFlightDay(from: Int, to: Int,
                               optNights: Option[Int],
                               optWhen: Option[LocalDate]): Directive1[FlightDay] = {
    val default = DefaultRequestGenerator.getDefaultFlightDay
    (optNights, optWhen) match {
      case (Some(nights), Some(when)) => provide(FlightDay(when, nights))
      case (Some(nights), None) => provide(FlightDay(default.when, nights))
      case (None, Some(when)) => provide(FlightDay(when, default.nights))
      case (None, None) =>
        calendarService match {
          case Some(calendar) =>
            val now = LocalDate.now
            val when = default.when
            val interval = DateInterval(now, now.plusYears(1))

            val day = calendar.getNearestFlightDay(from, to, when, interval).fallbackTo(Future.successful(None))
            onComplete(day).flatMap {
              case Success(Some(fd)) => provide(FlightDay(fd.getWhen.toLocalDate, fd.getNights))
              case _ => provide(default)
            }
          case None => provide(default)
        }
    }
  }
}
