package ru.yandex.tours.util.spray

import java.time.Month

import ru.yandex.tours.model.direction.Thematics
import ru.yandex.tours.model.search._
import ru.yandex.tours.model.util._
import ru.yandex.tours.util.parsing.IntValue
import shapeless.{::, HNil}
import spray.http.HttpHeader
import spray.routing.Directives._
import spray.routing._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.02.15
 */
trait SearchDirectives extends CommonDirectives with UserDirectives {

  def optDateInterval(name: String): Directive1[Option[DateInterval]] = {
    optDate(name) & parameter((name + "_flex").as[Boolean] ? false) &
      optDate(name + "_start") & optDate(name + "_end") hflatMap {
      case Some(when) :: flexWhen :: _ :: _ :: HNil =>
        provide(Some(FlexDateInterval(when, flexWhen)))
      case None :: _ :: Some(whenStart) :: Some(whenEnd) :: HNil =>
        provide(Some(CustomDateInterval(whenStart, whenEnd)))
      case _ => provide(None)
    }
  }

  def dateInterval(name: String): Directive1[DateInterval] = {
    optDateInterval(name) flatMap {
      case Some(interval) => provide(interval)
      case None => reject(MissingQueryParamRejection(name))
    }
  }

  def nights: Directive1[Nights] = {
    intArray("nights") & parameter('nights_flex.as[Boolean] ? false) hflatMap {
      case Seq(night) :: flexNights :: HNil =>
        provide(FlexNights(night, flexNights))
      case nights :: _ :: HNil =>
        provide(CustomNights(nights))
    }
  }

  def searchDates: Directive1[SearchDates] = {
    (dateInterval("when") & nights & optDateInterval("when_back")) hflatMap {
      case whenTo :: nights :: whenBack :: HNil =>
        provide(SearchDates(whenTo, nights, whenBack))
    }
  }

  val userRegion: Directive1[Option[Int]] = {
    parameter('user_region_id.as[Int].?) | optionalHeaderValuePF {
      case HttpHeader("X-UserRegionId", IntValue(rid)) => rid
    }
  }

  val whereToGoRequest: Directive1[WhereToGoRequest] = {
    enum("thematic", Thematics.values).flatMap { thematic =>
      parameters('budget.as[Int].?, 'month.as[Int].?, 'no_visa.as[Boolean] ? false, 'country.as[Int].?) hflatMap {
        case budget :: month :: noVisa :: countryId :: HNil =>
          val country = countryId.filter(_ => thematic == Thematics.Ski)
          provide(WhereToGoRequest(thematic, budget, month.map(Month.of), noVisa, country))
      }
    }
  }

  val geoId = parameter('geo_id.as[Int])

  val airportId = parameter('airport_id)

  val branding = parameter('branding.?)

  val fromOrGeoId: Directive1[Int] = parameter('geo_id.as[Int].?, 'from.as[Int].?) hflatMap {
    case optGeoId :: optFrom :: HNil => optFrom.orElse(optGeoId) match {
      case Some(result) => provide(result)
      case None => reject(MalformedQueryParamRejection("from", "from or geo_id should be specified"))
    }
  }

  val toOrGeoId: Directive1[Int] = parameter('geo_id.as[Int].?, 'to.as[Int].?) hflatMap {
    case optGeoId :: optTo :: HNil => optTo.orElse(optGeoId) match {
      case Some(result) => provide(result)
      case None => reject(MalformedQueryParamRejection("to", "to or geo_id should be specified"))
    }
  }

}

object SearchDirectives extends SearchDirectives