package ru.yandex.tours.util.spray.stream

import akka.util.ByteString
import spray.http.{ContentType, HttpEntity}
import spray.httpx.marshalling.Marshaller

import scala.language.implicitConversions

/**
 * Renders some values to [[ByteString]] for compose [[spray.http.HttpEntity]].
 * Main difference from [[spray.httpx.marshalling.Marshaller]] is ability to
 * render many values into single [[ByteString]]
 * and therefore into single [[spray.http.HttpEntity]].
 * [[spray.httpx.marshalling.Marshaller]] always renders
 * single element to single [[spray.http.HttpEntity]].
 *
 * Typically used for chunked response.
 * It may provide header and footer of resulting HTTP response entity.
 *
 * @author dimas
 */
trait Renderer[T] {
  /** Target content type */
  def contentType: ContentType

  /** Renders give value to byte string */
  def render(value: T): ByteString

  /** Renders many values into byte string */
  def renderMany(values: Seq[T]): ByteString =
    values.foldLeft(ByteString.empty) {
      (x, y) => x.concat(render(y))
    }

  /** Header is placed on top of HTTP response entity */
  def header: ByteString = ByteString.empty

  /** Footer is placed on bottom of HTTP response entity */
  def footer: ByteString = ByteString.empty
}

object Renderer {

  implicit def renderer2marshaller[T](renderer: Renderer[T]): Marshaller[T] =
    Marshaller.of[T](renderer.contentType) {
      case (value, ct, ctx) =>
        val byteString = renderer.render(value)
        val entity = HttpEntity(ct, byteString)
        ctx.marshalTo(entity)
    }

  implicit def renderer2seqMarshaller[T](renderer: Renderer[T]): Marshaller[Seq[T]] =
    Marshaller.of[Seq[T]](renderer.contentType) {
      case (values, ct, ctx) =>
        val byteString = renderer.renderMany(values)
        val entity = HttpEntity(ct, byteString)
        ctx.marshalTo(entity)
    }
}
