package ru.yandex.tours.util.trie;

/**
 * Aho-Corasick fsm from M.Crochemore, W.Rutter "Text algorithms"
 * <pre><code>
function AC(Tree(P); text) : boolean;
{ Aho-Corasick multi-pattern matching }
{ uses the table Bord on Tree(P)	}
begin
    state := root;
    read(symbol);
    while symbol <> endmarker do begin
        while state defined and son(state, symbol) undefined do
            state := Bord[state];
        if state undefined then
            state := root
        else
            state := son(state, symbol);

        if state is terminal then return true;
        read(symbol);
    end;
    return false;
end;
 * </code></pre>
 * @author aherman
 */
public class AhoCorasickMachine {
    private final Trie trie;
    private TrieNode state;
    private final CharSequence string;
    private final boolean debugMode;
    private int position;

    public AhoCorasickMachine(Trie trie, CharSequence string, boolean debugMode) {
        this.trie = trie;
        this.string = string;
        this.debugMode = debugMode;
        this.state = trie.getRoot();
        this.position = 0;
    }

    public int getPosition() {
        return position;
    }

    public State findNext() {
        if (position >= string.length()) {
            return State.END;
        }
        while(position < string.length()) {
            char ch = string.charAt(position);
            position++;

            TrieNode currentState = state;
            TrieNode nextStateCandidate = trie.getSon(state, ch);
            while (currentState.isDefined() && !nextStateCandidate.isDefined()) {
                currentState = trie.getBord(currentState);
                nextStateCandidate = trie.getSon(currentState, ch);
            }
            if (!currentState.isDefined()) {
                changeState(trie.getRoot(), ch);
                if (!debugMode) {
                    return State.ROOT;
                }
            } else {
                changeState(nextStateCandidate, ch);
            }
            if (nextStateCandidate.isTerminal()) {
                break;
            }
        }
        if (state.isTerminal()) {
            return State.TERMINAL;
        }
        return State.NON_TERMINAL;
    }

    private void changeState(TrieNode newState, char ch) {
//        log.info(this.state.getStateId() + " -> " + newState.getStateId() + " [label = \"" + ch + "\" ];");
//        System.out.println(this.state.getStateId() + " -> " + newState.getStateId() + " [label = \"" + ch + "\" ];");
        this.state = newState;
    }

    public int getStateId() {
        if (state.isTerminal()) {
            return state.getStateId();
        }
        return -1;
    }

    public static enum State {
        ROOT,
        TERMINAL,
        NON_TERMINAL,
        END
    }
}
