package ru.yandex.tours.util.trie;

import it.unimi.dsi.fastutil.ints.IntArrayFIFOQueue;
import it.unimi.dsi.fastutil.ints.IntPriorityQueue;

import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;

/**
 * Compute failure function Board for tree Tree(P) (from M.Crochemore, W.Rutter "Text algorithms")
 * <code><pre>
procedure compute_Bord; { failure table on Tree(P) }
begin
{ father and son refer to the tree Tree(P) }
{ Bord is defined on nodes of Tree(P)	}
{ if "z" is node and "a" is letter, "za" is child of "z" with edge, market with letter "a" }
{ Bord is defined on nodes }
{ for node x: \x\ is depth of node in tree }
    set Bord[Root] as undefined;
    for all child of Root do Bord[child] := Root;
    for all nodes x of Tree(P), |x|>2, in bfs order do begin
        t := father(x);
        a := the letter such that x=son(t, a);
        z := Bord[t];
        while z defined and za not in Tree(P) do z := Bord[z];
        if za is in Tree(P) then Bord[x] := za else Bord[x] := epsilon;
        if za is terminal then mark x as terminal;
    end;
end.
 * </pre></code>
 *
 * @author aherman
 */
public class BordBuilder {
    public static void buildBord(TrieBuilder trieBuilder, StateGluingListener listener) {
        List<TrieNodeBuilder> nodes = trieBuilder.getNodes();
        IntPriorityQueue queue = new IntArrayFIFOQueue();
        TrieNodeBuilder father = trieBuilder.getRoot();
        int rootId = father.getId();
        Iterator<Character> edges = father.getEdges();
        while (edges.hasNext()) {
            // now father is root
            int firstSonId = father.getSon(edges.next());
            queue.enqueue(firstSonId);
        }
        do {
            edges = father.getEdges();
            while (edges.hasNext()) {
                char a = edges.next();
                TrieNodeBuilder son = nodes.get(father.getSon(a));
                updateQueue(queue, son);

                int zId = father.getBord();
                int zaId = TrieNode.UNDEFINED_ID;

                while (zId != TrieNode.UNDEFINED_ID) {
                    TrieNodeBuilder z = nodes.get(zId);
                    zaId = z.getSon(a);
                    if (zaId != TrieNode.UNDEFINED_ID) {
                        break;
                    }
                    zId = z.getBord();
                }
                if (zaId == TrieNode.UNDEFINED_ID) {
                    son.setBord(rootId);
                } else {
                    son.setBord(zaId);
                    TrieNodeBuilder za = nodes.get(zaId);
                    if (za.isTerminal()) {
                        son.setTerminal(true);
                        if (listener != null) {
                            listener.statesAreGlued(son.getId(), zaId);
                        }
                    }
                }
            }
            if (queue.size() > 0) {
                int nextId = queue.dequeueInt();
                father = nodes.get(nextId);
            } else {
                father = null;
            }
        } while (father != null);
        optimizeBord(trieBuilder);
    }

    private static void optimizeBord(TrieBuilder trieBuilder) {
        IntPriorityQueue queue = new IntArrayFIFOQueue();
        queue.enqueue(trieBuilder.getRoot().getId());
        TrieNodeBuilder current;
        List<TrieNodeBuilder> nodes = trieBuilder.getNodes();
        while (queue.size() > 0) {
            current = nodes.get(queue.dequeueInt());
            updateQueue(queue, current);
            int bordId = current.getBord();
            while (bordId != TrieNode.UNDEFINED_ID) {
                TrieNodeBuilder bord = nodes.get(bordId);
                if (bord.getEdgesNumber() > 0) {
                    break;
                }
                bordId = bord.getBord();
            }
            if (bordId != TrieNode.UNDEFINED_ID) {
                current.setBord(bordId);
            }
        }
    }

    private static void updateQueue(IntPriorityQueue queue, TrieNodeBuilder node) {
        Iterator<Character> edges = node.getEdges();
        while (edges.hasNext()) {
            Character edge = edges.next();
            queue.enqueue(node.getSon(edge));
        }
    }

    public interface StateGluingListener {
        void statesAreGlued(int fromState, int toState);
    }
}
