package ru.yandex.tours.util.trie;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * @author aherman
 */
public class TrieBuilder {
    private static final int ROOT_NODE_ID = 0;
    private final ArrayList<TrieNodeBuilder> nodes;

    public TrieBuilder() {
        nodes = new ArrayList<TrieNodeBuilder>(16);
        TrieNodeBuilder root = new TrieNodeBuilder(ROOT_NODE_ID);
        nodes.add(root);
    }

    public int addString(CharSequence string) {
        TrieNodeBuilder currentNode = getRoot();
        for (int i = 0; i < string.length(); i++) {
            char ch = string.charAt(i);
            int sonId = currentNode.getSon(ch);
            if (sonId == TrieNode.UNDEFINED_ID) {
                TrieNodeBuilder son = new TrieNodeBuilder(nodes.size());
                nodes.add(son);
                currentNode.addSon(ch, son.getId());
                currentNode = son;
            } else {
                currentNode = nodes.get(sonId);
            }
        }
        currentNode.setTerminal(true);
        return currentNode.getId();
    }

    public ArrayTrie build() {
        TrieNode[] optimizedNodes = new TrieNode[nodes.size()];
        for (int i = 0; i < nodes.size(); i++) {
            TrieNode node = nodes.get(i).build();
            optimizedNodes[i] = node;
        }
        return new ArrayTrie(optimizedNodes);
    }

    TrieNodeBuilder getRoot() {
        return nodes.get(ROOT_NODE_ID);
    }

    List<TrieNodeBuilder> getNodes() {
        return nodes;
    }

    public String toString() {
        return toDot();
    }

    public String toDot() {
        StringBuilder sb = new StringBuilder();
        sb.append("digraph fsm").append('{').append('\n');
        sb.append("node [shape=circle, width=.3, height=.3, fontsize = 10, fixedsize = true, fontname=\"Monospaces\"]").append('\n');
        sb.append("rankdir = LR;").append('\n');
        printNode(getRoot(), sb, "");
        sb.append('}').append('\n');
        return sb.toString();
    }

    private void printNode(TrieNodeBuilder node, StringBuilder sb, String indent) {
        if (node.isTerminal()) {
            sb.append(node.getId()).append("[style = filled, fillcolor = black, fontcolor = white]").append(
                    ";\n");
        }
        if (node.getBord() != TrieNode.UNDEFINED_ID && node.getBord() != 0) {
            sb.append(String.format("%d -> %d [ style = dashed, constraint = true ];", node.getId(), node.getBord())).append('\n');
        }
        Iterator<Character> edges = node.getEdges();
        while(edges.hasNext()) {
            Character edge = edges.next();
            int sonId = node.getSon(edge);
            sb.append(String.format("%d -> %d [ label = \"%c\" ];", node.getId(), sonId, edge)).append('\n');
            TrieNodeBuilder son = nodes.get(sonId);
            printNode(son, sb, "");
        }
    }
}
