package ru.yandex.tours.util.trie;

import java.util.Iterator;

/**
 * @author aherman
 */
public abstract class TrieNode {
    public static int UNDEFINED_ID = -1;

    private final int stateId;
    private final boolean terminal;
    private final int bord;

    public TrieNode(int stateId, boolean terminal, int bord) {
        this.stateId = stateId;
        this.terminal = terminal;
        this.bord = bord;
    }


    public abstract int getSon(char ch);

    public int getBord() {
        return bord;
    }

    public boolean isTerminal() {
        return terminal;
    }

    public boolean isDefined() {
        return true;
    }

    public int getStateId() {
        return stateId;
    }

    public static class ManyChildNode extends TrieNode {
        private final FastCharMap edges;
        private final int bloomFilter;

        public ManyChildNode(int stateId, boolean terminal, int bord, FastCharMap edges) {
            super(stateId, terminal, bord);
            this.edges = edges;
            int bloomFilter = 0;
            if (edges != null && !edges.isEmpty()) {
                Iterator<Character> keys = this.edges.getKeys();
                while (keys.hasNext()) {
                    Character ch = keys.next();
                    bloomFilter |= ch;
                }
            }
            this.bloomFilter = bloomFilter;
        }

        public FastCharMap getEdges() {
            return edges;
        }

        public int getBloomFilter() {
            return bloomFilter;
        }

        @Override
        public int getSon(char ch) {
            if ((this.bloomFilter & ch) != ch) {
                return UNDEFINED_ID;
            }
            return edges.get(ch, UNDEFINED_ID);
        }
    }

    public static TrieNode UNDEFINED = new TrieNode(UNDEFINED_ID, false, UNDEFINED_ID) {
        @Override
        public int getSon(char ch) {
            return UNDEFINED_ID;
        }

        @Override
        public boolean isDefined() {
            return false;
        }
    };
}
