package ru.yandex.tours.util.trie.mapped

import java.nio.ByteBuffer
import java.util

import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.trie.AbstractFastCharMap

import scala.annotation.tailrec

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 05.10.15
 */
class MappedFastCharMap(buffer: ByteBuffer, val size: Int) extends AbstractFastCharMap with Logging {

  private def charAt(position: Int): Char = buffer.getChar(position * Character.BYTES)
  private def valueAt(position: Int): Int = buffer.getInt(size * Character.BYTES + position * Integer.BYTES)

  override def get(ch: Char, defaultValue: Int): Int = {
    @tailrec
    def binSearch(start: Int, end: Int): Int = {
      if (start > end) return start
      val position = (end + start) / 2

      val cmp = Character.compare(charAt(position), ch)

      if (cmp < 0) binSearch(position + 1, end)
      else if (cmp > 0) binSearch(start, position - 1)
      else position
    }

    val pos = binSearch(0, size - 1)
    if (pos >= size) defaultValue
    else {
      if (charAt(pos) == ch) valueAt(pos)
      else defaultValue
    }
  }

  override def put(ch: Char, i: Int): Unit = throw new UnsupportedOperationException

  override def getKeys: util.Iterator[Character] = {
    new util.Iterator[Character] {
      private var i = 0
      override def hasNext: Boolean = i < size
      override def next(): Character = {
        val c = charAt(i)
        i += 1
        c
      }
    }
  }

  override def isEmpty: Boolean = size == 0
}
