package ru.yandex.tours.util.trie.mapped

import java.io.OutputStream
import java.nio.ByteBuffer

import ru.yandex.tours.util.collections.MappedArray
import ru.yandex.tours.util.io.ByteBuffers
import ru.yandex.tours.util.trie.{ArrayTrie, Trie, TrieNode}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 05.10.15
 */
class MappedTrie(buffer: ByteBuffer) extends Trie {

  private val arr = MappedArray[TrieNode](buffer)(MappedTrieNode.fromBuffer)

  val size = arr.size

  private def getNode(id: Int): TrieNode = {
    require(id < size, s"Node id must be in range [0, $size): $id")
    arr.get(id)
  }

  override def getRoot: TrieNode = getNode(Trie.ROOT_ID)

  override def getSon(node: TrieNode, ch: Char): TrieNode = {
    if (!node.isDefined) {
      return TrieNode.UNDEFINED
    }
    val sonId = node.getSon(ch)
    if (sonId == TrieNode.UNDEFINED_ID) {
      return TrieNode.UNDEFINED
    }
    getNode(sonId)
  }

  override def getBord(node: TrieNode): TrieNode = {
    if (!node.isDefined) {
      return TrieNode.UNDEFINED
    }
    val bordId: Int = node.getBord
    if (bordId == TrieNode.UNDEFINED_ID) {
      return TrieNode.UNDEFINED
    }
    getNode(bordId)
  }

  override def toString: String = s"MappedTrie(size = $size)"

  override def writeTo(os: OutputStream): Unit = ByteBuffers.write(buffer, os)
}

object MappedTrie {
  def writeTo(os: OutputStream, trie: ArrayTrie): Unit = {
    val nodes = trie.getNodes.toSeq.asInstanceOf[Seq[TrieNode.ManyChildNode]]
    MappedArray.writeArray[TrieNode.ManyChildNode](nodes.toArray, None, MappedTrieNode.toBytes, os)
  }
}