package ru.yandex.tours.util.trie.mapped

import java.io.DataOutputStream
import java.nio.ByteBuffer

import org.apache.commons.io.output.ByteArrayOutputStream
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.trie.TrieNode

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 05.10.15
 */
class MappedTrieNode(stateId: Int, terminal: Boolean, bord: Int, bloomFilter: Int,
                     mapBuffer: ByteBuffer, mapSize: Int)
  extends TrieNode(stateId, terminal, bord) {

  override def getSon(ch: Char): Int = {
    if ((this.bloomFilter & ch) != ch) TrieNode.UNDEFINED_ID
    else {
      val map = new MappedFastCharMap(mapBuffer, mapSize)
      map.get(ch, TrieNode.UNDEFINED_ID)
    }
  }
}

object MappedTrieNode {
  def fromBuffer(buffer: ByteBuffer): MappedTrieNode = {
    val stateId = buffer.getInt
    val terminal = buffer.get() > 0
    val bord = buffer.getInt()
    val bloomFilter = buffer.getInt()
    val mapSize = buffer.getInt
    val map = buffer.slice()
    new MappedTrieNode(stateId, terminal, bord, bloomFilter, map, mapSize)
  }

  def toBytes(node: TrieNode.ManyChildNode): Array[Byte] = {
    val baos = new ByteArrayOutputStream()
    IO.using(new DataOutputStream(baos)) { os =>
      os.writeInt(node.getStateId)
      os.writeByte(if (node.isTerminal) 1 else 0)
      os.writeInt(node.getBord)
      os.writeInt(node.getBloomFilter)
      os.writeInt(node.getEdges.size())
      node.getEdges.getKeys.asScala.foreach(ch => os.writeChar(ch.toInt))
      node.getEdges.getValues.asScala.foreach(v => os.writeInt(v))
    }
    baos.toByteArray
  }
}