package ru.yandex.tours.util.zoo

import java.io.Closeable

import org.apache.curator.framework.CuratorFramework
import org.apache.curator.framework.recipes.AfterConnectionEstablished
import org.apache.curator.framework.recipes.shared.{SharedValue => CuratorSharedValue, SharedValueListener, SharedValueReader}
import org.apache.curator.framework.state.ConnectionState
import ru.yandex.vertis.curator.recipes.map.ValueSerializer

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 03.03.15
 */
class SharedValue[T](client: CuratorFramework, path: String, initialValue: T, serializer: ValueSerializer[T]) extends Closeable {

  private val shared = new CuratorSharedValue(client, path, serializer.serialize(initialValue))

  private var listeners = List.empty[T => Unit]

  shared.getListenable.addListener(new SharedValueListener {
    override def valueHasChanged(sharedValue: SharedValueReader, newBytes: Array[Byte]): Unit = {
      for (newValue <- serializer.deserialize(newBytes)) {
        listeners.foreach(_.apply(newValue))
      }
    }
    override def stateChanged(client: CuratorFramework, newState: ConnectionState): Unit = ()
  })

  AfterConnectionEstablished.execute(client, new Runnable {
    override def run(): Unit = {
      shared.start()
      for (newValue <- serializer.deserialize(shared.getValue)) {
        listeners.foreach(_.apply(newValue))
      }
    }
  })

  def set(newValue: T): Unit = {
    shared.setValue(serializer.serialize(newValue))
  }

  def get: T = serializer.deserialize(shared.getValue).getOrElse(initialValue)

  def onUpdate(action: T => Unit): Unit = {
    listeners ::= action
  }

  override def close(): Unit = shared.close()
}
