package ru.yandex.tours

import akka.actor.Props
import akka.routing.RoundRobinPool
import ru.yandex.tours.agencies._
import ru.yandex.tours.app._
import ru.yandex.tours.avatars._
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.db.{HotelsDbSupport, MySqlSupport}
import ru.yandex.tours.extdata.{DataTypes, ExtDataSupport}
import ru.yandex.tours.extdataloader.ExtDataLoaderServices
import ru.yandex.tours.extdataloader.verba.Verba
import ru.yandex.tours.geo.partners.PartnerTreesSupport
import ru.yandex.tours.hotels.clustering.ClusteringModel
import ru.yandex.tours.hotels.enrichers._
import ru.yandex.tours.indexer.IndexerHandler
import ru.yandex.tours.indexer.agencies.AgenciesIndexer
import ru.yandex.tours.indexer.airports.AirportsIndexer
import ru.yandex.tours.indexer.backa.{BackaExportHolder, BackaPermalinkIndexer}
import ru.yandex.tours.indexer.billing.{AgencyPartneringReceiver, BillingReceiver}
import ru.yandex.tours.indexer.booking.BookingHotelRegionsIndexer
import ru.yandex.tours.indexer.clusterization._
import ru.yandex.tours.indexer.currency.CurrencyRateDownloader
import ru.yandex.tours.indexer.data.{HydraSinkApi, SearchEventStreams}
import ru.yandex.tours.indexer.export.{BackaExporter, BackaMatchingExporter}
import ru.yandex.tours.indexer.geomapping.{GeoMappingIndexer, PartnerTreesIndexer, PartnersGeoMatcher}
import ru.yandex.tours.indexer.google.GooglePlacesIndexer
import ru.yandex.tours.indexer.hotelcounts.PartnerHotelCountsIndexer
import ru.yandex.tours.indexer.hotels._
import ru.yandex.tours.indexer.hotels.parsers.booking.HttpHotel2RegionRetriever
import ru.yandex.tours.indexer.hotelscombined.HCId2KeyIndexer
import ru.yandex.tours.indexer.ml.HotelsTfIdfIndexer
import ru.yandex.tours.indexer.ota.{OtaHotelIndexer, OtaOperatorIndexer, OtaRegionIndexer}
import ru.yandex.tours.indexer.ratings.PartnerRatingsIndexer
import ru.yandex.tours.indexer.regions.GeobasePatcher
import ru.yandex.tours.indexer.searchmapping.SearchSettingsIndexer
import ru.yandex.tours.indexer.sitemap.SitemapIndexer
import ru.yandex.tours.indexer.suggest.SuggestIndexer
import ru.yandex.tours.partners.HotelsCombinedId2Key
import ru.yandex.tours.metro.MetroHolder
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.ota.OnlineTourAgencies
import ru.yandex.tours.partners.PartnerConfigs
import ru.yandex.tours.util.file._
import ru.yandex.tours.util.naming.HotelsTfIdfModel
import ru.yandex.vertis.scheduler.model.Task
import shapeless.HNil

import scala.concurrent.duration._
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 19.02.15
 */
trait IndexerComponents extends SchedulerTasks {
  app: Application
    with AkkaSupport with SpraySupport
    with CommonDataHolders
    with PartnerTreesSupport
    with ExtDataSupport with ExtDataLoaderServices
    with HttpSupport with ZookeeperSupport
    with MySqlSupport
    with SchedulerSupport
    with S3Support
    with AviaClientSupport
    with HotelsDbSupport
    with GeoIdSettersSupport
    with AvatarSupport =>

  override def lazyLoadResources: Boolean = true

  lazy val verba = Verba.from(extDataService, extDataUpdateChecker, lazyLoad = true)
  lazy val backaPermalinks = BackaPermalinks.from(extDataService, extDataUpdateChecker, lazyLoadResources)
  lazy val hcId2Key = HotelsCombinedId2Key.from(extDataService, extDataUpdateChecker, lazyLoadResources)

  lazy val metroHolder = MetroHolder.from(extDataService, extDataUpdateChecker, lazyLoad = true)
  lazy val agenciesBillingIndex = AgenciesBillingIndex.from(HNil, extDataService, extDataUpdateChecker, lazyLoad = true)
  lazy val agenciesPartneringIndex = AgencyPartneringIndex.from(extDataService, extDataUpdateChecker, lazyLoad = true)
  lazy val agencies = Agencies.from(extDataService, extDataUpdateChecker, lazyLoad = true)
  lazy val agenciesIndex = AgenciesIndex.from(agencies :: regionTree :: agenciesBillingIndex :: agenciesPartneringIndex :: CustomerToOperatorIndex.empty :: HNil, extDataUpdateChecker, lazyLoad = true)
  lazy val onlineAgencies = OnlineTourAgencies.from(extDataService, extDataUpdateChecker, lazyLoad = true)
  lazy val airportsIndexer = new AirportsIndexer(aviaClient, regionTree, dataPersistenceManager, 1.hours)(akkaSystem.dispatcher)

  lazy val geobasePatcher = new GeobasePatcher(db, regionBoundaries, extDataService, dataPersistenceManager)(akkaSystem.dispatcher)
  lazy val searchMappingIndexer = new SearchSettingsIndexer(db, dataPersistenceManager)(akkaSystem.dispatcher)
  lazy val backaExportHolder = new BackaExportHolder(config, app.dataFolder / "data", httpClient)(akkaSystem.dispatcher)
  lazy val agencyIndexer = new AgenciesIndexer(backaExportHolder, regionTree, dataPersistenceManager, backaExportHolder.getUpdateTime(), extDataService, metroHolder)(akkaSystem.dispatcher)
  lazy val agencyBillingReceiver = new BillingReceiver(extDataService, dataPersistenceManager, 1.minute, DataTypes.offlineAgencyBilling)(akkaSystem.dispatcher)
  lazy val hotelBillingReceiver = new BillingReceiver(extDataService, dataPersistenceManager, 1.minute, DataTypes.hotelBilling)(akkaSystem.dispatcher)
  lazy val cmHotelBillingReceiver = new BillingReceiver(extDataService, dataPersistenceManager, 1.minute, DataTypes.cmHotelBilling)(akkaSystem.dispatcher)
  lazy val agencyPartneringReceiver = new AgencyPartneringReceiver(extDataService, dataPersistenceManager, 10.minute)(akkaSystem.dispatcher)
  lazy val otaHotelIndexer = new OtaHotelIndexer(onlineAgencies, hotelsIndex, 12.hours, externalHttpClient, extDataService, dataPersistenceManager)(akkaSystem.dispatcher)
  lazy val otaRegionIndexer = new OtaRegionIndexer(onlineAgencies, oldLtGeoMappingHolder, 12.hours, externalHttpClient, extDataService, dataPersistenceManager)(akkaSystem.dispatcher)
  lazy val otaOperatorIndexer = new OtaOperatorIndexer(onlineAgencies, tourOperators, 12.hours, externalHttpClient, extDataService, dataPersistenceManager)(akkaSystem.dispatcher)
  lazy val sitemapIndexer = new SitemapIndexer(hotelsIndex, geoMappingHolder, skiResorts, s3Client, 12.hours)(akkaSystem.dispatcher)
  lazy val suggestIndexer = new SuggestIndexer(hotelsIndex, geoMappingHolder, regionTree, countryPriorities, resortPriorities, hotelPriorities, searchSettings, extDataService, s3Client, 12.hours)(akkaSystem.dispatcher)
  lazy val hotelsExporter = new HotelsExporter(hotelsIndex, regionTree, dataPersistenceManager, 12.hours)(akkaSystem.dispatcher)
  lazy val partnerHotelCountsIndexer = new PartnerHotelCountsIndexer(dataPersistenceManager, hotelsIndex, 24.hours)(akkaSystem.dispatcher)
  lazy val currencyRateDownloader = new CurrencyRateDownloader(externalHttpClient, dataPersistenceManager, 1.hour)(akkaSystem.dispatcher)

  lazy val geoMappingIndexerTypes = List(DataTypes.cities, DataTypes.countries, DataTypes.departures)
  lazy val geoMappingIndexer = new GeoMappingIndexer(geoMappingIndexerTypes, db, regionTree, airports, dataPersistenceManager, 10.minutes)(akkaSystem.dispatcher)
  lazy val partnerTreesIndexer = new PartnerTreesIndexer(extDataService, dataPersistenceManager, 12.hours)(akkaSystem.dispatcher)
  lazy val partnersGeoMatcher = new PartnersGeoMatcher(regionTree, partnerTrees, hotelsIndex, app.config.getConfig("geo.matching"), db, extDataService, 12.hours, dataPersistenceManager)(akkaSystem.dispatcher)

  lazy val hiddenFiltersCalculator = new HiddenFiltersCalculator(regionTree)
  lazy val hiddenFiltersIndexer = new HiddenFiltersIndexer(hiddenFiltersCalculator, dataPersistenceManager, hotelsIndex, 12.hours)(akkaSystem.dispatcher)
  lazy val hotelsTfIdfIndexer = new HotelsTfIdfIndexer(hotelsIndex, dataPersistenceManager, 2.hours)

  lazy val hotelsIndexer = new HotelsIndexer(db, hotelsDao, metaGeoIdSetter, regionTree, hotelRatings, backaPermalinks,
    dataPersistenceManager, 48.hours)(akkaSystem.dispatcher, akkaSystem)

  lazy val googlePlacesIndexer = new GooglePlacesIndexer(db, hotelsDao, httpClient, 24.hours)(akkaSystem.dispatcher, akkaSystem)
  lazy val httpBookingHotelRegionRetrievier = new HttpHotel2RegionRetriever(externalHttpClient,
    PartnerConfigs.parse(config)(Partners.booking))(akkaSystem.dispatcher)
  lazy val bookingHotelRegionIndexer = new BookingHotelRegionsIndexer(httpBookingHotelRegionRetrievier, s3Client, 12.hours)(akkaSystem.dispatcher)
  lazy val partnerRatingsIndexer = new PartnerRatingsIndexer(hotelsDao, dataPersistenceManager, 12.hours)(akkaSystem.dispatcher)
  lazy val backaMatchingExporter = new BackaMatchingExporter(hotelsIndex, s3Client, urlBuilder, 12.hours)(akkaSystem.dispatcher)
  lazy val backaExporter = new BackaExporter(hotelsIndex, regionTree, s3Client, urlBuilder, geoMappingHolder, searchSettings, hcId2Key, 12.hours)(akkaSystem.dispatcher)
  lazy val hcId2KeyIndexer = new HCId2KeyIndexer(dataPersistenceManager, hotelsDao, 24.hours)(akkaSystem.dispatcher)
  lazy val backaPermalinkIndexer = new BackaPermalinkIndexer(backaExportHolder, dataPersistenceManager,
    backaExportHolder.getUpdateTime(), extDataService)(akkaSystem.dispatcher)

  lazy val hotelClusterizer2 = {
    val ec = akkaSystem.dispatchers.lookup("akka.actor.hotels-clustering-dispatcher")
    val tfIdfModel = HotelsTfIdfModel.from(extDataService, extDataUpdateChecker, lazyLoad = true)
    val clusteringModel = ClusteringModel.from(tfIdfModel :: HNil, extDataService, extDataUpdateChecker)

    new NewHotelClusterizer(db, hotelsDao, clusteringModel, updateTime = 24.hours)(akkaSystem, ec)
  }

  /*lazy val hotelsCombinedClusterizer = {
    val ec = akkaSystem.dispatchers.lookup("akka.actor.hotels-clustering-dispatcher")
    val tfIdfModel = HotelsTfIdfModel.from(extDataService, extDataUpdateChecker, lazyLoad = true)
    val clusteringModel = ClusteringModel.from(tfIdfModel :: HNil, extDataService, extDataUpdateChecker)

    new HotelsCombinedClusterizer(db, hotelsDao, clusteringModel)(akkaSystem, ec)
  }*/

  lazy val ostrovokv3Clusterizer = {
    val ec = akkaSystem.dispatchers.lookup("akka.actor.hotels-clustering-dispatcher")
    val tfIdfModel = HotelsTfIdfModel.from(extDataService, extDataUpdateChecker, lazyLoad = true)
    val clusteringModel = ClusteringModel.from(tfIdfModel :: HNil, extDataService, extDataUpdateChecker)

    new OstrovokV3Clusterizer(db, hotelsDao, clusteringModel)(akkaSystem, ec)
  }

  lazy val ostrovokv3ClusterizerByOldId = {
    val ec = akkaSystem.dispatchers.lookup("akka.actor.hotels-clustering-dispatcher")
    val tfIdfModel = HotelsTfIdfModel.from(extDataService, extDataUpdateChecker, lazyLoad = true)
    val clusteringModel = ClusteringModel.from(tfIdfModel :: HNil, extDataService, extDataUpdateChecker)

    new OstrovokV3ClusterizerByOldId(db, hotelsDao, clusteringModel)(akkaSystem, ec)
  }

  lazy val dataLoaders = new IndexerDataLoaders(
    app.dataFolder / "data",
    app.config,
    avatarClient,
    verba,
    hotelsIndex,
    backaPermalinks,
    geoMappingHolder,
    regionTree,
    directions,
    extDataService
  )(akkaSystem.dispatcher)

  private val onlyStableTasks = if (isEnvironmentStable) Seq(googlePlacesIndexer) else Seq()

  override def schedulingTasks: Seq[Task] = super.schedulingTasks ++ (Seq(
    geobasePatcher,
    searchMappingIndexer,
    backaExportHolder,
    agencyIndexer,
    agencyBillingReceiver,
    hotelBillingReceiver,
    cmHotelBillingReceiver,
    otaHotelIndexer,
    otaRegionIndexer,
    otaOperatorIndexer,
    sitemapIndexer,
    suggestIndexer,
    hotelsExporter,
    airportsIndexer,
    geoMappingIndexer,
    partnerTreesIndexer,
    partnersGeoMatcher,
    hiddenFiltersIndexer,
    hotelsIndexer,
    partnerHotelCountsIndexer,
    hotelClusterizer2,
    //hotelsCombinedClusterizer, // manual task, remove after HC integration
    ostrovokv3Clusterizer, // manual task, remove after Ostrovok integration
    ostrovokv3ClusterizerByOldId, // manual task, remove after Ostrovok integration
    partnerRatingsIndexer,
    bookingHotelRegionIndexer,
    backaMatchingExporter,
    currencyRateDownloader,
    hotelsTfIdfIndexer,
    backaExporter,
    hcId2KeyIndexer,
    backaPermalinkIndexer
  ) ++ onlyStableTasks).map(_.asTask)

  val agenciesHandler = akkaSystem.actorOf(Props(new AgenciesHandler(agencies, agenciesIndex, metroHolder)), "agencies-handler")

  lazy val searchEventStreams = new SearchEventStreams(akkaSystem, dataFolder / "search_events")
  lazy val hydraSinkApi = new HydraSinkApi(searchEventStreams)(akkaSystem.dispatcher)

  val handlerProps = {
    Props(new IndexerHandler(metaDataStorageService, extDataStorageService, extDataLoaderEngine, avatarClient,
      agencyBillingReceiver, hotelBillingReceiver, cmHotelBillingReceiver, agencyPartneringReceiver, tourOperators,
      agenciesHandler, scheduler.getApi, hydraSinkApi))
      .withRouter(RoundRobinPool(nrOfInstances = 16))
      .withDispatcher("akka.actor.handler-dispatcher")
  }

  onStart {
    startServer(handlerProps, config.getInt("component.port"))
  }
}
