package ru.yandex.tours

import java.io.File

import com.typesafe.config.Config
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.tours.agencies.Agencies
import ru.yandex.tours.avatars.AvatarClient
import ru.yandex.tours.avia.{Airports, AviaAirportRecommendations, AviaCities}
import ru.yandex.tours.backa.BackaPermalinks
import ru.yandex.tours.currency.CurrencyRates
import ru.yandex.tours.direction._
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.extdataloader.DataLoaders
import ru.yandex.tours.extdataloader.verba.Verba
import ru.yandex.tours.extdataloader.verba.parsers._
import ru.yandex.tours.filter.HiddenFilters
import ru.yandex.tours.geo.base.region.{RegionBoundaries, Tree}
import ru.yandex.tours.geo.base.{GeoSynonyms, region}
import ru.yandex.tours.geo.mapping._
import ru.yandex.tours.geo.partners.{PartnerRegionHotelCounts, PartnerRegionsDef}
import ru.yandex.tours.geo.{Departures, Iso2Country, LtActiveCountries}
import ru.yandex.tours.hotels._
import ru.yandex.tours.indexer.hotels.Slopes
import ru.yandex.tours.indexer.metro.MetroJsonParser
import ru.yandex.tours.indexer.ratings.RatingsCombiner
import ru.yandex.tours.indexer.resorts.SkiResortsProcessor
import ru.yandex.tours.operators.{HotelProviders, TourOperators}
import ru.yandex.tours.parsing.PansionUnifier
import ru.yandex.tours.resorts.{Brandings, SkiResorts}
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.tanker.Translations
import ru.yandex.tours.util.naming.TfIdfModel
import shapeless.HNil

import scala.concurrent.ExecutionContext

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.02.15
 */
class IndexerDataLoaders(dataDir: File,
                         config: Config,
                         avatarClient: AvatarClient,
                         verba: Verba,
                         hotelsIndex: HotelsIndex,
                         backaPermalinks: BackaPermalinks,
                         geoMappingHolder: GeoMappingHolder,
                         tree: region.Tree,
                         directions: Directions,
                         extDataService: ExtDataService)
                        (implicit ec: ExecutionContext) extends DataLoaders {

  DataTypes.jordan
    .loadFromS3(config)

  DataTypes.geobase
    .loadFromHttp(config.getConfig("geobase"))
    .validateWith(Tree.parse)

  DataTypes.regions
    .provided(versionsToStore = 3)
    .validateWith(Tree.parse)

  DataTypes.searchSettings
    .provided(versionsToStore = 1)
    .validateWith(SearchSettingsHolder.parse(_, Tree.empty :: LtActiveCountries.empty:: HNil))

  DataTypes.brandings
    .loadFromS3(config)
    .validateWith(Brandings.parse)

  DataTypes.tanker
    .loadFromHttp(config.getConfig("tanker"))
    .validateWith(Translations.parse)

  DataTypes.verba
    .loadFromHttp(config.getConfig("verba"))
    .validateWith(Verba.parse)

  DataTypes.departureStats
    .loadFromS3(config)
    .validateWith(Departures.parse(_, Tree.empty :: GeoMappingHolder.empty:: HNil))
  DataTypes.geoSimilarity
    .loadFromS3(config)
    .validateWith(DirectionsSimilarity.parse)
  DataTypes.hotelsSimilarity
    .loadFromS3(config)
    .validateWith(HotelsSimilarity.parse(_, HotelsIndex.empty :: HNil))

  //geoMappings are loaded from mysql
  DataTypes.cities.provided(versionsToStore = 3).validateWith(CityMapping.parse)
  DataTypes.countries.provided(versionsToStore = 3).validateWith(CountryMapping.parse)
  DataTypes.departures.provided(versionsToStore = 3).validateWith(DepartureMapping.parse)
  DataTypes.airports.loadFromS3(config).validateWith(AirportMapping.parse)

  DataTypes.ltActiveCountries.loadFromS3(config, customName = Some("lt.active.countries.tsv"))

  DataTypes.regionBoundaries.loadFromS3(config).validateWith(RegionBoundaries.parse)
  DataTypes.iso2country.loadFromS3(config).validateWith(Iso2Country.parse(_, Tree.empty :: HNil))

  DataTypes.oldLtCities.loadFromS3(config).validateWith(OldLtCityMapping.parse)
  DataTypes.oldLtCountries.loadFromS3(config).validateWith(OldLtCountryMapping.parse)
  DataTypes.oldLtDepartures.loadFromS3(config).validateWith(OldLtDepartureMapping.parse)

  DataTypes.agenciesProto.provided(versionsToStore = 3).validateWith(Agencies.parse)

  DataTypes.operators.combine(
    new TourOperatorsBuilder(verba).buildAndSerialize("tour_operators", priorityDictionary = ""),
    DataTypes.verba
  ).validateWith(TourOperators.parse)

  DataTypes.hotel_providers.combine(
    new TourOperatorsBuilder(verba)
      .buildAndSerialize("hotel_partners", priorityDictionary = "hotel_provider_priorities"),
    DataTypes.verba
  ).validateWith(HotelProviders.parse)

  DataTypes.hotelsVideo.combine(
    new HotelsVideoBuilder(verba, avatarClient).buildAndSerialize(),
    DataTypes.verba
  ).validateWith(HotelsVideo.parse)

  DataTypes.hotelsPanoramas
    .loadFromS3(config)
    .validateWith(HotelsPanoramas.parse(_, hotelsIndex :: HNil))

  DataTypes.hotelUGCRatings
    .loadFromS3(config)

  DataTypes.hotelPartnerRatings
    .provided(versionsToStore = 1)

  DataTypes.backaPermalinks
    .provided(versionsToStore = 1)

  //todo add validation
  DataTypes.hotelsVisits
    .loadFromS3(config)

  DataTypes.hotelsRatings
    .combine(new RatingsCombiner(hotelsIndex, backaPermalinks, extDataService).build)
    .depends(DataTypes.hotelUGCRatings, DataTypes.hotelPartnerRatings, DataTypes.hotelsVisits)
    .depends(DataTypes.shardedHotels.dataTypes: _*)
    .validateWith(HotelRatings.parse)

  DataTypes.aviaAirports.provided(2).validateWith(Airports.parse)
  DataTypes.aviaCities.provided(2).validateWith(AviaCities.parse)
  DataTypes.airportRecommendations
    .loadFromS3(config)
    .validateWith(AviaAirportRecommendations.parse(_, Tree.empty :: AviaCities.empty :: HNil))

  // todo validation
  DataTypes.otaCitiesMappings.provided(versionsToStore = 3)
  DataTypes.otaHotelMappings.provided(versionsToStore = 3)
  DataTypes.otaOperatorMappings.provided(versionsToStore = 3)
  DataTypes.hotelsExport.provided(versionsToStore = 2)

  // todo validate
  DataTypes.partnerConversion
    .combine(new PartnerStatisticParser(verba, "partner_conversions").build(), DataTypes.verba)

  // todo validation
  DataTypes.directions.combine(
    new DirectionsBuilder(avatarClient, verba, tree).buildAndSerialize(),
    DataTypes.regions, DataTypes.verba
  )

  DataTypes.pansions.combine(
    new PansionAliasesBuilder(verba).buildAndSerialize(),
    DataTypes.verba
  ).validateWith(PansionUnifier.parse)

  DataTypes.currencyRates
    .provided(versionsToStore = 1)
    .validateWith(CurrencyRates.parse)

  DataTypes.hiddenFilters.provided(3).validateWith(HiddenFilters.parse)

  DataTypes.geoSynonyms.loadFromS3(config).validateWith(GeoSynonyms.parse)
  DataTypes.countryPriorities.loadFromS3(config).validateWith(CountryPriorities.parse)
  DataTypes.hotelPriorities.loadFromS3(config).validateWith(HotelPriorities.parse)
  DataTypes.resortPriorities.loadFromS3(config).validateWith(ResortPriorities.parse)

  // todo validate
  DataTypes.offlineAgencyBilling.provided(versionsToStore = 3)
  DataTypes.hotelBilling.provided(versionsToStore = 3)
  DataTypes.cmHotelBilling.provided(versionsToStore = 3)
  DataTypes.offlineAgencyPartnering.provided(versionsToStore = 3)
  DataTypes.onlineAgencies.combine(new OnlineAgenciesParser(verba, "online_tour_agencies").build(), DataTypes.verba)

  // todo validation
  DataTypes.substitutePhones.combine(new PhonesParser(verba, "agency_substitution_phones").build(), DataTypes.verba)
  DataTypes.customerToOperator.combine(new CustomerToOperatorParser(verba, "customer_to_tour_operator").build(), DataTypes.verba)
  DataTypes.agenciesShows.loadFromS3(config)
  DataTypes.metro.loadFromHttpWithProcessing(config.getConfig("metro"), MetroJsonParser.jsonIs2is)

  // todo validation
  DataTypes.shardedHotels.dataTypes.foreach(_.provided(versionsToStore = 2))

  DataTypes.hotelsTfIdfModel.provided(versionsToStore = 2).validateWith(TfIdfModel.parse)

  // todo validation
  DataTypes.clusteringMatrixnetModel.loadFromS3(config)

  DataTypes.geoSuggest.provided(versionsToStore = 2)

  DataTypes.wizardIndex.dataTypes.foreach(_.provided(versionsToStore = 3))
  DataTypes.shardedMicroOffers.dataTypes.foreach(_.provided(versionsToStore = 3))

  DataTypes.wizardTrie.provided(2)
  DataTypes.wizardReqAnsTrie.provided(2)
  DataTypes.stopWordsTrie.provided(2)

  DataTypes.wizardReqAns.loadFromS3(config)

  // todo validation
  DataTypes.dates.combine(new WizardDatesParser(verba, "wizard.dates").build(), DataTypes.verba)
  DataTypes.stopWords.combine(new WizardPhrasesParser(verba, "wizard.stop_words").build(), DataTypes.verba)
  DataTypes.ignoreWords.combine(new WizardPhrasesParser(verba, "wizard.ignore_words").build(), DataTypes.verba)
  DataTypes.wizardMarkers.combine(new WizardMarkersParser(verba, "wizard.markers").build(), DataTypes.verba)
  DataTypes.wizardGeo.combine(new WizardGeoParser(verba, "wizard.geo", tree, geoMappingHolder).build(), DataTypes.verba, DataTypes.regions)
  DataTypes.operatorAliases.combine(new WizardOperatorsParser(verba, "tour_operators").build(), DataTypes.verba)
  DataTypes.hotelNameParts.provided(versionsToStore = 1)

  DataTypes.skiResorts.loadFromS3WithProcessing(
    config,
    customConfigPath = Some("ski_resorts"),
    customName = Some("ski_resorts.version5.json"),
    processor = new SkiResortsProcessor(avatarClient = avatarClient, directions = directions).process
  ).validateWith(SkiResorts.parse)

  DataTypes.slopes
    .loadFromS3(config)
    .validateWith(Slopes.parse)

  DataTypes.partnerRegions
    .loadFromS3(config, customName = Some("partner_regions_v2.tsv"))
    .validateWith(PartnerRegionsDef.parse(_, PartnerRegionHotelCounts.empty :: HNil))

  DataTypes.partnerTrees.values.foreach(_.provided(versionsToStore = 2))

  // todo validation
  DataTypes.geoMatchingHypotheses.provided(1)
  DataTypes.partnerRegionHotelCounts.provided(1)
  DataTypes.hcId2Key.provided(1)
  DataTypes.hotDirections.loadFromS3(config).validateWith(HotDirections.parse)
}
