package ru.yandex.tours

import ru.yandex.tours.app._
import ru.yandex.tours.extdata.ExtDataSupport
import ru.yandex.tours.extdataloader.ExtDataLoaderServices
import ru.yandex.tours.extdataloader.verba.parsers.{WizardHotelAliasesReader, WizardHotelNamePartsIndexer}
import ru.yandex.tours.index.WizardIndexing.MicroOffers
import ru.yandex.tours.index.{IndexCleaner, WizardIndexing}
import ru.yandex.tours.indexer.wizard._
import ru.yandex.tours.indexer.wizard.parser.ParsingTrieTasks
import ru.yandex.tours.query.parser.{LocalParserResources, ParserResources}
import ru.yandex.tours.util.file._
import ru.yandex.vertis.scheduler.model.Task

import scala.collection.Map
import scala.concurrent.duration._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 13.03.15
 */
trait WizardIndexingSupport extends SchedulerTasks {
  app: Application
    with AkkaSupport
    with ZookeeperSupport
    with CommonDataHolders with ExtDataSupport
    with ExtDataLoaderServices
    with IndexerComponents =>

  lazy val parserResources: ParserResources = new LocalParserResources(extDataService) {
    override def getAliases: Map[Int, Seq[String]] = new WizardHotelAliasesReader(verba).loadAliases
  }
  lazy val hotelNamePartsIndexer = new WizardHotelNamePartsIndexer(() => parserResources.getAliases, hotelsIndex, 1.hour, dataPersistenceManager)
  lazy val trieTasks = new ParsingTrieTasks(parserResources, dataPersistenceManager)

  private val freshIndexDir = app.dataFolder / "index"

  private lazy val distributor = new ResourceDistributor(WizardIndexing.topic, zkClient, WizardIndexing.zkPath)(akkaSystem)

  private lazy val indexBuilder = new IndexBuilder(regionTree, hotelsIndex, geoMappingHolder)

  private lazy val indexer = new Indexer(indexBuilder, freshIndexDir)

  lazy val wizardScheduler = new IndexScheduler(indexer, distributor, dataPersistenceManager)

  private val freshMicroOfferDir = app.dataFolder / "micro_offers"
  private lazy val microOfferBuilder = new MicroOfferBuilder(hotelsIndex)
  private lazy val microOfferIndexer = new MicroOfferIndexer(microOfferBuilder, freshMicroOfferDir)
  private lazy val moDistributor = new ResourceDistributor(MicroOffers.topic, zkClient, WizardIndexing.zkPath)(akkaSystem)
  private lazy val microOfferScheduler = new MicroOfferScheduler(microOfferIndexer, moDistributor, dataPersistenceManager)

  override def schedulingTasks: Seq[Task] = super.schedulingTasks ++ Seq(
    wizardScheduler.freshIndexTask,
    wizardScheduler.longIndexTask,
    trieTasks.buildStopWordsTask,
    trieTasks.buildWizardTrieTask,
    trieTasks.buildWizardReqAnsTrieTask,
    hotelNamePartsIndexer.asTask,
    microOfferScheduler.freshOffersTask,
    microOfferScheduler.longOffersTask
  )

  onStart {
    IndexCleaner.start(freshIndexDir, WizardIndexing.indexExtension, ttl = 24.hours, interval = 1.hour)(akkaSystem)
    IndexCleaner.start(freshMicroOfferDir, MicroOffers.extension, ttl = 24.hours, interval = 1.hour)(akkaSystem)
    WizardSearchEventReceiver(searchEventStreams)(indexer, microOfferIndexer)(akkaSystem.dispatcher)
  }

}
