package ru.yandex.tours.avatars

import play.api.libs.json.Json
import ru.yandex.tours.model.Image
import ru.yandex.tours.model.image.ImageProviders.ImageProvider
import ru.yandex.tours.util.Logging
import spray.http.{StatusCode, StatusCodes}

import scala.concurrent.{ExecutionContext, Future}
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 28.04.15
 */
class CachingAvatarClient(dao: AvatarDao, client: AvatarClient,
                          table: String = "avatars")
                         (implicit ec: ExecutionContext) extends AvatarClient with Logging {

  override def put(url: String, provider: ImageProvider): Future[Image] = {
    dao.getImage(url).flatMap {
      case Some(image) =>
        recover(image)
      case None =>
        for {
          image <- client.put(url, provider)
          _ <- dao.putImage(url, image)
        } yield image
    }
  }

  override def putRaw(url: String, imageName: String, provider: ImageProvider): Future[(StatusCode, String)] = {
    put(url, provider).map { image =>
      val json = Json.obj("group-id" -> image.group)
      StatusCodes.OK -> json.toString()
    }.recover {
      case NonFatal(t) =>
        log.warn("Failed to upload image", t)
        StatusCodes.InternalServerError -> t.getMessage
    }
  }


  override def recover(image: Image): Future[Image] = {
    if (!image.isNeedRecover) return Future.successful(image)
    for {
      recovered <- client.recover(image)
      _ <- dao.saveSizeAndPHash(recovered)
    } yield recovered
  }
}
