package ru.yandex.tours.avatars

import java.io.ByteArrayInputStream

import akka.util.Timeout
import org.json.JSONObject
import ru.yandex.tours.model.Image
import ru.yandex.tours.model.Image.Size
import ru.yandex.tours.model.image.ImageFormats
import ru.yandex.tours.model.image.ImageProviders._
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.{ImageUtils, Logging}
import spray.http.{StatusCode, StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.control.NonFatal

class HttpAvatarClient(client: AsyncHttpClient, writeUrl: String, readUrl: String, namespace: String)
                      (implicit ec: ExecutionContext)
  extends AvatarClient with Logging {

  private implicit val timeout = Timeout(60.seconds)

  def put(url: String, provider: ImageProvider): Future[Image] = {
    val imageName = Image.name(url)
    putRaw(url, imageName, provider).map {
      case (StatusCodes.OK, entity) =>
        val json = new JSONObject(entity)
        val groupId = json.getInt("group-id")
        val meta = json.getJSONObject("meta")
        val nnFeatures = parseNNFeatures(meta)
        val size = parseImageSize(meta)
        Image(readUrl,
          groupId,
          provider,
          imageName,
          Some(url),
          size = size,
          pHash = None,
          nnFeatures = nnFeatures)
      case (sc, entity) =>
        throw new Exception(s"Avatar client responded non 200: $sc, $entity, $url, $imageName")
    }.flatMap(recover)
  }

  def putRaw(url: String, imageName: String, provider: ImageProvider): Future[(StatusCode, String)] = {
    val requestUrl = Uri(s"$writeUrl/put-$namespace/$imageName")
      .withQuery("url" -> url, "provider" -> provider.toString)
    client.get(requestUrl)
  }

  private def parseImageSize(json: JSONObject): Option[Size] = {
    if (json.has("orig-size")) {
      val origSize = json.getJSONObject("orig-size")
      Some(Image.Size(origSize.getInt("x"), origSize.getInt("y")))
    } else {
      None
    }

  }

  private def parseNNFeatures(json: JSONObject): Option[Array[Float]] = {
    if (json.has("NNetFeatures")) {
      Some(json.getJSONObject("NNetFeatures").getString("fc_encoder_96").split(" ").map(_.toFloat))
    } else {
      None
    }
  }

  private def getMeta(image: Image): Future[Option[JSONObject]] = {
    val uri = Uri(s"$writeUrl/getinfo-tours/${image.group}/${image.name}/meta")
    client.get(uri).map {
      case (StatusCodes.OK, body) =>
        Some(new JSONObject(body))
      case (sc, entity) =>
        throw new Exception(s"Avatar client responded non 200: $sc, $entity, $uri, ${image.name}")
    }.recover {
      case NonFatal(t) =>
        log.debug("Failed to load image meta", t)
        None
    }
  }

  private def getImageHash(image: Image): Future[Option[String]] = {
    if (image.pHash.nonEmpty) return Future.successful(image.pHash)
    val uri = image.inFormat(ImageFormats.small)
    client.getBytes(uri).map {
      case (StatusCodes.OK, bytes) =>
        Some(ImageUtils.pHash(new ByteArrayInputStream(bytes)))
      case (sc, entity) =>
        throw new Exception(s"Avatar client responded non 200: $sc, $entity, $uri, ${image.name}")
    }.recover {
      case NonFatal(t) =>
        log.debug("Failed to calculate image pHash", t)
        None
    }
  }

  override def recover(image: Image): Future[Image] = {
    if (!image.isNeedRecover) return Future.successful(image)

    for {
      meta <- getMeta(image)
      pHash <- getImageHash(image)
    } yield {
      val size = meta.flatMap(parseImageSize)
      val nnFeatures = meta.flatMap(parseNNFeatures)
      image.copy(size = size, pHash = pHash, nnFeatures = nnFeatures)
    }
  }
}

