package ru.yandex.tours.avatars

import ru.yandex.tours.avatars.yt.Avatar
import ru.yandex.tours.model.Image
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.travel.yt.{Factory, YtDao}

import scala.concurrent.{ExecutionContext, Future}

class YtAvatarDao(dao: YtDao[Avatar])
                 (implicit ec: ExecutionContext) extends AvatarDao with Logging {

  override def getImage(url: String): Future[Option[Image]] = {
    log.debug(s"Getting image $url")
   for (oa <- dao.get(url).asScalaFuture) yield
     for (a <- oa.asOption) yield {
       val host = a.getHost
       val group = a.getGroup
       val provider = ImageProviders(a.getProvider)
       val name = a.getName
       val source = Option(a.getUrl)
       val size =
         if (a.getWidth == null || a.getHeight == null) None
         else Some(Image.Size(a.getWidth, a.getHeight))
       val pHash = if (a.getpHash()==null) None else Option(a.getpHash())
       val nnFeatures = if (a.getNNF == null) None else Option(a.getNNF.split(" ").map(_.toFloat))
       Image(host, group, provider, name, source, size, pHash, nnFeatures)
     }
  }

  override def putImage(url: String, image: Image): Future[Image] = {
    log.debug(s"Putting image $url")
    dao.put(new Avatar(url, Int.box(image.group), Int.box(image.provider.id), image.host, image.name,
      image.size.map(_.width).map(Int.box).orNull,
      image.size.map(_.height).map(Int.box).orNull,
      image.pHash.orNull,
      image.nnFeatures.map(_.mkString(" ")).orNull)).asScalaFuture.toUnit.map(_ => image)
  }

  override def saveSizeAndPHash(image: Image): Future[Unit] = {
    log.debug(s"Updating image ${image.source.orNull}")
    dao.get(image.source.orNull).asScalaFuture.map(optAv => {
      optAv.asOption.foreach(a => {
        a.setWidth(image.size.map(_.width).map(Int.box).orNull)
        a.setHeight(image.size.map(_.height).map(Int.box).orNull)
        a.setpHash(image.pHash.orNull)
        a.setNNF(image.nnFeatures.map(_.mkString(" ")).orNull)
        dao.put(a).asScalaFuture.toUnit
      })
    })
  }
}
