package ru.yandex.tours.export

import scala.collection.immutable.HashSet
import java.io.{Closeable, PrintWriter}
import java.text.{DecimalFormat, DecimalFormatSymbols}

import ru.yandex.common.util.StringEscapeUtils
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Features._
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.image.{ImageFormats, ImageProviders}
import ru.yandex.tours.model.image.ImageFormats.ImageFormat
import ru.yandex.tours.model.search.SearchType
import ru.yandex.tours.model.utm.UtmMark
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.json.CommonJsonSerialization.breadcrumbsToJson
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.util.Logging

/* @author berkut@yandex-team.ru */
/* Despite of name, this exporter is also used for full hotels export to YT */

class BackaFormatExporter(output: PrintWriter,
                          urlBuilder: UrlBuilder,
                          tree: region.Tree,
                          geoMapping: GeoMappingHolder,
                          searchSettings: SearchSettingsHolder,
                          hcId2Key: Map[String, String]) extends Logging with Closeable {
  private val myFormatter = {
    val format = new DecimalFormat("###.#####")
    val dfs = format.getDecimalFormatSymbols
    dfs.setDecimalSeparator('.')
    format.setDecimalFormatSymbols(dfs)
    format
  }
  private var exported: Int = 0

  start()

  private def start() {
    output.println("<?xml version=\"1.0\" encoding=\"utf-8\"?><companies>")
  }

  private def end() {
    output.println("</companies>")
    output.close()
    log.info("Success export " + exported + " hotels in backa format")
  }

  def add(hotel: Hotel, extraFields: Boolean = false) {
    output.print("<company>")
    output.print(s"<company-id>${hotel.id}</company-id>")
    for ((lang, name) ← hotel.name.values) {
      addLangTag("name", lang, name)
    }
    for (address <- hotel.addresses.values) {

      if (address.country.nonEmpty) {
        addLangTag("country", address.lang, address.country)
      }
      if (address.locality.nonEmpty) {
        addLangTag("locality-name", address.lang, address.locality)
      }
      if (address.fullAddress.nonEmpty) {
        addLangTag("address", address.lang, address.fullAddress)
      }
    }
    output.print(s"<coordinates><lat>${myFormatter.format(hotel.latitude)}</lat>")
    output.print(s"<lon>${myFormatter.format(hotel.longitude)}</lon></coordinates>")
    addPhone(hotel)
    hotel.url.foreach { url =>
      output.print(s"<url>${escape(url)}</url>")
    }
    val url = "https:" + urlBuilder.forHotelDefault(hotel.id, UtmMark(Some("yandex_maps")), SearchType.ROOMS)
    output.print(s"<add-url>${escape(url)}</add-url>")
    output.print("<rubric-id>184106414</rubric-id>")
    output.print(s"<actualization-date>${System.currentTimeMillis()}</actualization-date>")
    if (hotel.star.id > 0) {
      output.print(s"""<feature-enum-single name="star" value="${escape(hotel.star.name)}"/>""")
    }
    hotel.features.foreach {
      case BooleanFeatureValue(BooleanFeature(name), Some(value)) =>
        val b = if (value) "1" else "0"
        output.print(s"""<feature-boolean name="$name" value="$b"/>""")
      case IntFeatureValue(IntFeature(name), Some(value)) =>
        output.print(s"""<feature-numeric-single name="$name" value="$value"/>""")
      case EnumFeatureValue(EnumFeature(name, _), Some(value)) =>
        output.print(s"""<feature-enum-single name="$name" value="${escape(value)}"/>""")
      case MultipleFeatureValue(MultipleEnumFeature(name, _), values) =>
        for (value <- values) {
          output.print(s"""<feature-enum-multiple name="$name" value="${escape(value)}"/>""")
        }
      case feature ⇒
        log.warn(s"Unexpected feature: $feature")
    }
    addAdditionalInfo(hotel, extraFields)
    addImages(hotel, extraFields)
    output.println("</company>")
    exported += 1
  }

  private def escape(url: String): String = {
    StringEscapeUtils.escapeXml(url)
  }

  private def addPhone(hotel: Hotel) {
    for {
      phone <- hotel.phones
      if phone.nonEmpty
    } {
      output.print("<phone>")
      output.print(s"<number>${escape(phone)}</number>")
      output.print("<ext/>")
      output.print("<info/>")
      output.print("<type>phone</type>")
      output.print("</phone>")
    }
  }

  private def addLangTag(name: String, lang: Languages.Lang, value: String) = {
    output.print(s"""<$name lang="$lang">${escape(value)}</$name>""")
  }

  private def addAdditionalInfo(hotel: Hotel, extraFields: Boolean) {
    output.print("<add-info>")
    for (pid <- hotel.partnerIds) {
      output.print(s"""<partner name="${escape(pid.partner.toString)}">""")
      if (extraFields) {
        output.print(s"<travel-id>${pid.travelId}</travel-id>")
        if (pid.partner == Partners.hotelsCombined) {
          output.print(s"<key>${hcId2Key.get(pid.id).getOrElse("")}</key>")
        }
      }
      output.print(s"<id>${escape(pid.id)}</id>")
      output.print(s"<url>${escape(pid.url)}</url>")
      output.print(s"</partner>")
    }
    if (extraFields) {
      if (hotel.backaPermalink.isDefined) {
        output.print(s"<backa-permalink>${hotel.backaPermalink.get}</backa-permalink>")
      }
      output.print(s"<tours-search>${hotel.toursSearchAvailable}</tours-search>")
      output.print(s"<rooms-search>${hotel.roomsSearchAvailable}</rooms-search>")
      output.print(s"<rating>${hotel.rating}</rating>")
      output.print(s"<reviews>${hotel.reviewsCount}</reviews>")
      output.print(s"<geo-id>${hotel.geoId}</geo-id>")
      output.print(s"<type>${hotel.`type`.toString}</type>")
      for (lang <- Set(Languages.en, Languages.ru)) {
        val regions = breadcrumbsToJson(hotel.geoId, tree, geoMapping, searchSettings, lang)
        addLangTag("region", lang, regions.toString())
      }
    }
    output.print("</add-info>")
  }

  private val BannedImageProviders = HashSet(ImageProviders.oktogo, ImageProviders.ya, ImageProviders.lih)

  private def addImages(hotel: Hotel, extraFields: Boolean) {
    output.print("<photos>")
    for (image <- hotel.images) {
      if (extraFields || !BannedImageProviders.contains(image.provider)) {
        output.print(s"""<photo url="${escape(image.inFormat(ImageFormats.orig))}" provider="${escape(image.provider.toString)}"/>""")
      }
    }
    output.print("</photos>")
  }

  def close(): Unit = {
    end()
    output.close()
  }

}
