package ru.yandex.tours.extdataloader

import java.io.{ByteArrayInputStream, File, InputStream}
import java.util

import com.typesafe.config.Config
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.loader.loaders.DependentLoader.HashCodeCollector
import ru.yandex.extdata.loader.loaders.{DataLoader, DependentLoader, DirectHttpLoader, LocalFsLoader}
import ru.yandex.extdata.loader.specs.{DataLoadSpec, DependentDataLoadSpec, HttpDataLoadSpec, LocalFileLoadSpec}
import ru.yandex.extdata.loader.validators.DataValidator
import ru.yandex.tours.extdataloader.verba.{Verba, VerbaDataLoadSpec, VerbaLoader}
import ru.yandex.tours.util.file._
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.verba2.model.Service

import scala.collection.JavaConverters._
import scala.collection.mutable
import scala.concurrent.duration.FiniteDuration

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 10.03.15
 */
trait DataLoaders {
  private val loadSpecs = mutable.HashMap.empty[DataType, DataLoadSpec]
  private val dataLoaders = mutable.ListBuffer.empty[DataLoader[_ <: DataLoadSpec]]
  private val dataValidators = mutable.HashMap.empty[DataType, List[DataValidator[_]]]

  implicit class RichDataType(dt: DataType) {

    def loadFromHttp(url: String, interval: FiniteDuration, versionsToStore: Int) = {
      loadSpecs += dt -> new HttpDataLoadSpec(
        dt,
        interval.toSeconds.toInt,
        versionsToStore,
        url
      )
      dataLoaders += {
        val loader = new DirectHttpLoader()
        loader.setDataType(dt)
        loader
      }
      dt
    }
    
    def loadFromHttpWithProcessing(url: String, interval: FiniteDuration, versionsToStore: Int, processor: InputStream => InputStream) = {
      loadSpecs += dt -> new HttpDataLoadSpec(
        dt,
        interval.toSeconds.toInt,
        versionsToStore,
        url
      )
      dataLoaders += {
        val loader = new HttpDataWithProcessingLoader(processor)
        loader.setDataType(dt)
        loader
      }
      dt
    }

    def loadFromHttpWithProcessing(conf: Config, processor: InputStream => InputStream): DataType = {
      loadFromHttpWithProcessing(
        conf.getString("url"),
        conf.getFiniteDuration("interval"),
        conf.getInt("versions"),
        processor
      )
    }

    def loadFromHttp(conf: Config): DataType = {
      loadFromHttp(
        conf.getString("url"),
        conf.getFiniteDuration("interval"),
        conf.getInt("versions")
      )
    }

    def loadFromS3(conf: Config,
                   customConfigPath: Option[String] = None,
                   customName: Option[String] = None): DataType = {
      val s3Url = conf.getString("tours.s3.url")
      val url = s"$s3Url/${customName.getOrElse(dt.getName)}"
      val config = conf.getConfig(customConfigPath.getOrElse(dt.getName))
      loadFromHttp(
        url,
        config.getFiniteDuration("interval"),
        config.getInt("versions")
      )
    }

    def loadFromS3WithProcessing(conf: Config,
                                 customConfigPath: Option[String] = None,
                                 customName: Option[String] = None,
                                 processor: InputStream => InputStream): DataType = {
      val s3Url = conf.getString("tours.s3.url")
      val url = s"$s3Url/${customName.getOrElse(dt.getName)}"
      val config = conf.getConfig(customConfigPath.getOrElse(dt.getName))
      loadFromHttpWithProcessing(
        url,
        config.getFiniteDuration("interval"),
        config.getInt("versions"),
        processor
      )
    }

    def combine(builder: => Array[Byte], dependencies: DataType*): DataType = {
      loadSpecs += dt -> new DependentDataLoadSpec(dt, -1, 1)
      dataLoaders += {
        val loader = new DependentLoader {
          override def buildData(hashCodeCollector: HashCodeCollector): InputStream = {
            new ByteArrayInputStream(builder)
          }
        }
        loader.setDataType(dt)
        loader
      }
      dt.depends(dependencies: _*)
    }

    def depends(dependencies: DataType*): DataType = {
      for {
        dep <- dependencies
        loader <- loaders.find(_.getDataType == dep)
      } {
        val newDependencies = new util.ArrayList[DataType](loader.getDependentTypes)
        newDependencies.add(dt)
        loader.setDependentTypes(newDependencies)
      }
      dt
    }

    def provided[T <: DataLoadSpec](versionsToStore: Int): DataType = {
      loadSpecs += dt -> new ProvidedDataLoadSpec(dt, versionsToStore)
      dt
    }

    def validateWith[T](parser: InputStream => T): DataType = {
      val oldValidators = dataValidators.getOrElse(dt, List.empty)
      val validator = new DataValidator[Array[Byte]] {
        override def validateData(bytes: Array[Byte]): Unit = parser(new ByteArrayInputStream(bytes))
      }
      dataValidators += dt -> (validator :: oldValidators)
      dt
    }
  }

  def specs: util.Map[DataType, DataLoadSpec] = loadSpecs.asJava
  def loadableSpecs: util.Map[DataType, DataLoadSpec] =
    loadSpecs.filterNot(_._2.isInstanceOf[ProvidedDataLoadSpec]).asJava
  def loaders: List[DataLoader[_ <: DataLoadSpec]] = dataLoaders.toList
  def validators: util.Map[DataType, util.List[DataValidator[_]]] = dataValidators.mapValues(_.asJava).asJava
}
