package ru.yandex.tours.extdataloader

import java.util.Collections

import ru.yandex.extdata.loader.engine.{DataPersistenceManager, ExtDataLoaderEngine}
import ru.yandex.extdata.loader.loaders._
import ru.yandex.extdata.loader.storage.{DataStorageService, MetaDataStorageService}
import ru.yandex.extdata.loader.validators.ValidatorService
import ru.yandex.tours.app.Application
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.extdataloader.verba.VerbaLoader
import ru.yandex.tours.util.file._

import scala.collection.JavaConverters._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.02.15
 */
trait ExtDataLoaderServices { app: Application =>
  def metaDataStorageService: MetaDataStorageService
  def extDataStorageService: DataStorageService

  def dataLoaders: DataLoaders

  private lazy val validatorService = {
    new ValidatorService(dataLoaders.validators, Collections.emptyList())
  }

  lazy val dataPersistenceManager: DataPersistenceManager = {
    val manager = new DataPersistenceManager
    manager.setDataStorage(extDataStorageService)
    manager.setMetaStorage(metaDataStorageService)
    manager.setValidatorService(validatorService)
    manager.setSpecs(dataLoaders.specs)
    manager
  }

  private lazy val dataChecker = new DataChecker(metaDataStorageService, dataPersistenceManager)
  private lazy val httpClient = new ThreadedHttpClient(6, 10000)

  private lazy val dataLoaderService = {
    val loaders = dataLoaders.loaders
    loaders.foreach(_.setChecker(dataChecker))
    loaders.foreach {
      case http: DirectHttpLoader =>
        http.setHttpLoader(httpClient)
      case verba: VerbaLoader =>
        verba.setHttpLoader(httpClient)
      case httpWithProcessor: HttpDataWithProcessingLoader =>
        httpWithProcessor.setHttpLoader(httpClient)
      case _ =>
    }
    loaders.foreach(_.afterPropertiesSet())
    new DataLoaderService(loaders.asJava)
  }

  lazy val extDataLoaderEngine: ExtDataLoaderEngine = {
    val engine = new ExtDataLoaderEngine(dataLoaders.loadableSpecs, dataLoaderService)
    engine.setDataPersistenceManager(dataPersistenceManager)
    engine.setMetaStorage(metaDataStorageService)
    engine.setDataStorage(extDataStorageService)
    engine.setDataTypeRegistry(DataTypes.registry)
    engine
  }

  onStart {
    extDataLoaderEngine.start()
  }

  onStop {
    httpClient.destroy()
    extDataLoaderEngine.stop()
  }
}

trait FSExtDataLoaderServices extends ExtDataLoaderServices { app: Application =>
  lazy val metaDataStorageService = new FSMetaDataStorageService(app.dataFolder / "ext-meta")
  lazy val extDataStorageService = new FSDataStorageService(app.dataFolder / "ext-data")
}