package ru.yandex.tours.extdataloader

import java.io.{File, FileInputStream, FileOutputStream}
import java.util.concurrent.locks.ReentrantReadWriteLock

import ru.yandex.extdata.common.message.ExtDataSchema
import ru.yandex.extdata.common.meta.MetaDataInstance
import ru.yandex.extdata.loader.storage.MetaDataStorageService
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.file._

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.02.15
 */
class FSMetaDataStorageService(dir: File) extends MetaDataStorageService {
  require(dir.exists() || dir.mkdirs(), "Cannot create directory " + dir)

  private val rwLock = new ReentrantReadWriteLock()

  override def storeMetaData(meta: MetaDataInstance): Unit = {
    val lock = rwLock.writeLock()
    lock.lock()
    try {
      IO.using(new FileOutputStream(dir / meta.getDataTypeName)) { os =>
        meta.toMessage.writeTo(os)
      }
    } finally lock.unlock()
  }

  override def getMetaData(typeName: String): MetaDataInstance = {
    val file = dir / typeName
    val lock = rwLock.readLock()
    lock.lock()
    try {
      if (!file.exists()) return null

      IO.using(new FileInputStream(file)) { is =>
        val proto = ExtDataSchema.ExtDataMetaInfoMessage.parseFrom(is)
        MetaDataInstance.fromMessage(proto)
      }
    } finally lock.unlock()
  }
}
