package ru.yandex.tours.extdataloader.verba.parsers

import java.io.PrintWriter
import java.time.Month

import org.apache.commons.codec.binary.Base64
import org.apache.commons.io.output.ByteArrayOutputStream
import ru.yandex.tours.avatars.AvatarClient
import ru.yandex.tours.direction.Direction
import ru.yandex.tours.extdataloader.verba.Verba
import ru.yandex.tours.extdataloader.verba.VerbaDSL._
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.model._
import ru.yandex.tours.model.direction.Thematics.Thematic
import ru.yandex.tours.model.direction.{ThematicInfo, Thematics}
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.util.lang.Futures
import ru.yandex.tours.util.{IO, Logging}
import ru.yandex.verba2.model.Term
import ru.yandex.verba2.model.attribute.ComplexAttribute

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContext, Future}
import scala.util.control.NonFatal

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 29.04.15
 */
class DirectionsBuilder(avatarClient: AvatarClient,
                        verba: Verba,
                        tree: Tree)(implicit ec: ExecutionContext) extends Logging {

  private val globalTimeout = 10.minutes
  private val dictionaryCode = "directions"

  private val seasonThematicOverride = Map(
    Thematics.Diving -> Thematics.Beach,
    Thematics.Surfing -> Thematics.Beach,
    Thematics.Exotic -> Thematics.Beach
  )

  case class RawDirection(geoId: Int,
                          images: Seq[String],
                          squareImages: Seq[String],
                          priority: Int,
                          noVisa: Boolean,
                          relevance: Map[Thematic, Double],
                          seasons: Map[Thematic, Seasonality],
                          highSeasons: Map[Thematic, Seasonality])

  private def parseImages(term: Term, attrName: String) = {
    for {
      imageAttr <- term.attr[ComplexAttribute](attrName).getComplexEntities
      image <- imageAttr.image("photo")
    } yield image.getUrl
  }

  private def parseSeasonality(term: Term, prefix: String) = {
    val intervals: Seq[Seasonality] = for {
      attr <- term.attrOpt[ComplexAttribute](prefix + "_season").toSeq
      intervalAttr <- attr.getComplexEntities
    } yield {
      val start = intervalAttr.link("start").getCode
      val end = intervalAttr.link("end").getCode
      (start, end) match {
        case ("all_year", "all_year") => AllYear
        case _ => MonthInterval(Month.valueOf(start.toUpperCase), Month.valueOf(end.toUpperCase))
      }
    }

    Seasonality(intervals: _*)
  }

  def parseRawDirections: Seq[RawDirection] = {
    val dictionary = verba.dictionary(dictionaryCode)

    for {
      term <- dictionary.getTerms
    } yield {
      val geoId = term.intValue("geo_id")
      val priority = term.intValue("priority")

      val normalImages = parseImages(term, "images")
      val squareImages = parseImages(term, "square_images")

      val noVisa = term.booleanOptValue("no_visa").getOrElse(false)

      val relevance = Thematics.values.toSeq.map { thematic =>
        thematic -> term.doubleOptValue(thematic.toString.toLowerCase).getOrElse(0d)
      }.toMap

      val seasons = Thematics.values.toSeq.map { thematic =>
        thematic -> parseSeasonality(term, thematic.toString.toLowerCase)
      }.toMap

      val highSeasons = Thematics.values.toSeq.map { thematic =>
        thematic -> parseSeasonality(term, thematic.toString.toLowerCase + "_hi")
      }.toMap

      RawDirection(geoId, normalImages, squareImages, priority, noVisa, relevance, seasons, highSeasons)
    }
  }

  def downloadImages(imageUrls: Seq[String], client: AvatarClient): Future[Map[String, Image]] = {
    val images = new mutable.HashMap[String, Future[Image]]()
    for (imageUrl <- imageUrls) {
      images(imageUrl) = client.put(imageUrl, ImageProviders.ya)
    }
    images.values.foreach(_.onFailure {
      case NonFatal(ex) => log.warn(s"Failed to load image: ${ex.getMessage}")
    })
    Futures.all(images.values).map { _ =>
      images.map { case (imageUrl, future) =>
        val image = Await.result(future, Duration.Inf) //result should be ready at this moment
        imageUrl -> image
      }(collection.breakOut)
    }
  }

  def buildDirections: Seq[Direction] = {
    val rawDirections = parseRawDirections
    log.info(s"Found ${rawDirections.size} raw directions")

    val images = Await.result(
      downloadImages(rawDirections.flatMap(d => d.images ++ d.squareImages).distinct, avatarClient),
      globalTimeout
    )
    log.info(s"Loaded ${images.size} images")

    val rawDirectionsMap = rawDirections.map(d => d.geoId -> d).toMap

    for {
      direction <- rawDirections
      region <- tree.region(direction.geoId)
    } yield {
      def find(cond: RawDirection => Boolean): Option[RawDirection] = {
        tree.pathToRoot(region).flatMap(r => rawDirectionsMap.get(r.id)).find(cond)
      }
      def getImages(imageOrigUrlsFor: RawDirection => Seq[String], map: Map[String, Image]): Seq[Image] = {
        if (imageOrigUrlsFor(direction).nonEmpty) {
          imageOrigUrlsFor(direction).flatMap(map.get)
        } else {
          tree.findChildren(region).toSeq
            .flatMap(r => rawDirectionsMap.get(r.id))
            .flatMap(imageOrigUrlsFor).flatMap(map.get)
        }
      }

      val rImages = getImages(_.images, images)
      val sqImages = getImages(_.squareImages, images)
      val thematics = for (thematic <- Thematics.values.toSeq) yield {
        val seasonThematic = seasonThematicOverride.getOrElse(thematic, thematic)
        ThematicInfo(
          thematic,
          direction.relevance(thematic),
          direction.seasons(seasonThematic),
          direction.highSeasons(seasonThematic)
        )
      }
      log.debug(s"Direction ${direction.geoId}: ${images.size} + ${sqImages.size} images, " +
        s"priority = ${direction.priority}, thematics = $thematics")
      Direction(
        region,
        rImages,
        sqImages,
        direction.priority > 0,
        find(_.noVisa).fold(false)(_.noVisa),
        thematics,
        Seq.empty
      )
    }
  }

  private def marshallImages(images: Iterable[Image]) = {
    val os = new ByteArrayOutputStream()
    for (image <- images) {
      image.toProto.writeDelimitedTo(os)
    }
    Base64.encodeBase64String(os.toByteArray)
  }

  def buildAndSerialize(): Array[Byte] = {
    val out = new ByteArrayOutputStream()
    IO.using(new PrintWriter(out)) { writer =>
      for (direction <- buildDirections) {
        if (direction.images.isEmpty || direction.squareImages.isEmpty) {
          log.info(s"Ignoring direction without images: ${direction.region.id} ${direction.region.name.ruName}")
        } else {
          val images = marshallImages(direction.images)
          val squareImages = marshallImages(direction.squareImages)
          val thematics = direction.thematics.map(ThematicInfo.serialize).mkString(";")
          val priority = if (direction.isPromoted) 1 else 0
          writer.println(s"${direction.region.id}\t$images\t$squareImages\t$priority\t${direction.noVisa}\t$thematics")
        }
      }
    }
    out.toByteArray
  }
}
