package ru.yandex.tours.extdataloader.verba.parsers

import java.io.PrintWriter

import org.apache.commons.io.output.ByteArrayOutputStream
import ru.yandex.tours.extdataloader.verba.VerbaDSL._
import ru.yandex.tours.extdataloader.verba.Verba
import ru.yandex.tours.model.TourOperator
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{AtSKV, Tabbed}
import ru.yandex.verba2.model.attribute.{AliasAttribute, ComplexAttribute}

import scala.collection.JavaConversions._
import scala.collection.mutable

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 08.05.15
 */
class TourOperatorsBuilder(verba: Verba) {

  def parseOperators(dictionaryCode: String): Seq[(Int, Map[Partner, String], String, String)] = {
    val dictionary = verba.dictionary(dictionaryCode)

    for {
      term <- dictionary.getTerms
    } yield {
      val id = term.intValue("id")
      val name = term.attr[AliasAttribute]("name").mainValue("ru")
      val code = term.stringValue("code")

      val mapping = for {
        mappingAttr <- term.attr[ComplexAttribute]("mapping").getComplexEntities
      } yield {
        val partnerId = mappingAttr.stringValue("partner_id")
        val operatorId = mappingAttr.stringValue("operator_id")
        Partners.withName(partnerId) -> operatorId
      }

      (id, mapping.toMap, name, code)
    }
  }

  def parsePriorities(dictionaryCode: String): Map[Int, Double] = {
    if (dictionaryCode.isEmpty) {
      Map.empty
    } else {
      val dictionary = verba.dictionary(dictionaryCode)

      val pairs = for (term ← dictionary.getTerms) yield {
        val id = term.intValue("id")
        val priority = term.doubleValue("priority")

        id → priority
      }
      pairs.toMap
    }
  }

  def buildAndSerialize(dictionaryCode: String, priorityDictionary: String): Array[Byte] = {
    val out = new ByteArrayOutputStream()
    val priorities = parsePriorities(priorityDictionary)
    IO.using(new PrintWriter(out)) { writer =>
      for ((id, mapping, name, code) <- parseOperators(dictionaryCode)) {
        writer.println(Tabbed(
          id,
          name,
          code,
          priorities.getOrElse(id, 30.0d),
          AtSKV(mapping.map(kv => kv._1.toString -> kv._2))
        ))
      }
    }
    out.toByteArray
  }
}
