package ru.yandex.tours.extdataloader.verba.parsers

import java.io.PrintWriter

import ru.yandex.tours.extdataloader.verba.Verba
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.base.region.Types
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.verba2.model.Dictionary
import ru.yandex.verba2.model.attribute.AliasAttribute

import scala.collection.JavaConversions._
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 29.04.15
 */
class WizardGeoParser(verba: Verba, dictionaryCode: String, tree: region.Tree, geoMappingHolder: GeoMappingHolder)
  extends VerbaDataParser(verba, dictionaryCode) {

  private val ignoredRegions = Set(11372, 109146, 112113, 3, 10838)
  private val regionTypes = Set(Types.Airport, Types.City, Types.Village, Types.Continent, Types.Country,
    Types.FederalDistrict, Types.FederalSubject, Types.FederalSubjectRegion, Types.Region)


  override protected def convert(dictionary: Dictionary, writer: PrintWriter): Unit = {
    val mapping = new mutable.HashMap[Int, ArrayBuffer[String]]()

    for {
      region <- tree.regions
      if regionTypes.contains(region.`type`) || geoMappingHolder.isKnownDestination(region.id) || geoMappingHolder.isDepartureCity(region.id)
      if !ignoredRegions.contains(region.id)
    } {
      var allNames = region.allNames.filter(_.trim.nonEmpty)
      if (region.genitive.nonEmpty) {
        allNames += region.genitive
        allNames += "в " + region.genitive
        allNames += "из " + region.genitive
        allNames += "по " + region.genitive
      }
      if (region.dative.nonEmpty) {
        allNames += region.dative
        allNames += "в " + region.dative
        allNames += "по " + region.dative
      }
      if (region.accusative.nonEmpty) {
        allNames += region.accusative
        allNames += "в " + region.accusative
        allNames += "во " + region.accusative
        allNames += "на " + region.accusative
      }
      if (region.locative.nonEmpty) {
        allNames += region.locative
        allNames += "в " + region.locative
        allNames += "во " + region.locative
        allNames += "на " + region.locative
      }

      mapping += (region.id -> allNames.to[ArrayBuffer])
    }

    for (term <- dictionary.getTerms) {
      val geoId = term.intValue("geo_id")
      val phrases = term.attr[AliasAttribute]("phrase").allValues("ru")

      val buffer = mapping.getOrElseUpdate(geoId, ArrayBuffer.empty[String])
      buffer ++= phrases
    }

    val reverseMapping = new mutable.HashMap[String, mutable.Set[Int]] with mutable.MultiMap[String, Int]
    for {
      (regionId, synonyms) <- mapping
      synonym <- synonyms
    } {
      reverseMapping.addBinding(synonym, regionId)
    }

    for ((name, regionIds) <- reverseMapping.toIterable) {
      val best = regionIds.maxBy(id => tree.region(id).get.position)
      writer.println(s"$best\t$name")
    }
  }
}
