package ru.yandex.tours.indexer

import akka.actor.ActorRef
import akka.stream.ActorMaterializer
import org.joda.time.Duration
import org.json.{JSONArray, JSONObject}
import ru.yandex.extdata.common.meta.MetaDataInstance
import ru.yandex.extdata.loader.engine.ExtDataLoaderEngine
import ru.yandex.extdata.loader.storage.{DataStorageService, MetaDataStorageService}
import ru.yandex.tours.avatars.AvatarClient
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.indexer.billing.{AgencyPartneringReceiver, BillingReceiver}
import ru.yandex.tours.indexer.data.HydraSinkApi
import ru.yandex.tours.model.image.ImageProviders
import ru.yandex.tours.operators.TourOperators
import ru.yandex.tours.util.spray._
import ru.yandex.tours.util.spray.stream.StreamingResponseSupport
import ru.yandex.tours.util.spray.stream.bytestring.ByteStringStreamingResponse
import ru.yandex.vertis.billing.Model.OfferBilling
import ru.yandex.vertis.scheduler.SchedulerApi
import shapeless._
import spray.http.StatusCodes
import spray.httpx.marshalling.MetaMarshallers
import spray.routing.{Directive1, Route}

import scala.concurrent.Future
import scala.util.{Failure, Success, Try}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 25.02.15
 */
class IndexerHandler(metaDataStorageService: MetaDataStorageService,
                     extDataStorageService: DataStorageService,
                     engine: ExtDataLoaderEngine,
                     avatarClient: AvatarClient,
                     agencyBillingReceiver: BillingReceiver,
                     hotelBillingReceiver: BillingReceiver,
                     cmHotelBillingReceiver: BillingReceiver,
                     agencyPartneringReceiver: AgencyPartneringReceiver,
                     tourOperators: TourOperators,
                     agencyHandler: ActorRef,
                     schedulerApi: SchedulerApi,
                     hydraSkinApi: HydraSinkApi)
  extends HttpHandler with CommonDirectives with StreamingResponseSupport with MetaMarshallers {

  private val metaData: Directive1[MetaDataInstance] = parameter("data-type") hflatMap {
    case typeName :: HNil =>
      metaDataStorageService.getMetaData(typeName) match {
        case null => complete(StatusCodes.NotFound, s"No meta-data for type $typeName")
        case meta => provide(meta)
      }
  }

  private def getDataInstance(meta: MetaDataInstance, formatVersion: Int, dataVersion: Option[Int]) = {
    dataVersion match {
      case None if meta.isDataLoaded(formatVersion) =>
        Option(meta.getLastDataInstance(formatVersion))
      case Some(version) if meta.isDataLoaded(formatVersion, version) =>
        Option(meta.getDataInstance(formatVersion, version))
      case _ =>
        Option.empty
    }
  }

  implicit val materializer = ActorMaterializer()

  lazy val route: Route =
    path("get-meta") {
      metaData { meta =>
        completeProto(meta.toMessage)
      }
    } ~ path("get-data") {
      (metaData & parameter("format-version".as[Int], "data-version".as[Int].?)) {
        (meta, formatVersion, dataVersion) =>
          val dataInstance = getDataInstance(meta, formatVersion, dataVersion)
          dataInstance match {
            case None => complete(StatusCodes.NotFound, s"No data for type ${meta.getDataTypeName}")
            case Some(instance) =>
              val dataId = instance.getDataId
              val is = extDataStorageService.readData(dataId)
              complete(new ByteStringStreamingResponse(is))
          }
      }
    } ~ path("reload") {
      parameter("data-type") { typeName =>
        val dataType = DataTypes.registry.byName(typeName)
        engine.load(dataType)
        complete(StatusCodes.Accepted, "data-reload-scheduled")
      }
    } ~ (path("put-tours" / Segment) & parameters('url) & enum("provider", ImageProviders.values)) {
      (imageName, url, provider) =>
        onSuccess(avatarClient.putRaw(url, imageName, provider)) {
          case (status, entity) => complete(status, entity)
        }
    } ~ pathPrefix("api" / "1.x") {
      path("billing" / "offer" / Segment) { rawBizId =>
        Try(if (rawBizId.startsWith("h")) {
          (rawBizId.substring(1).toLong, hotelBillingReceiver)
        } else if (rawBizId.startsWith("cmh")) {
          (rawBizId.substring(3).toLong, cmHotelBillingReceiver)
        } else {
          (rawBizId.toLong, agencyBillingReceiver)
        }) match {
          case Success((bizId, receiver)) =>
            ((put | post) & entity(as[Array[Byte]])) { rawOffer =>
              Try(OfferBilling.parseFrom(rawOffer)) match {
                case Success(offerBilling) =>
                  receiver.receive(bizId, offerBilling)
                  complete(StatusCodes.OK)
                case Failure(e) =>
                  complete(StatusCodes.BadRequest, "Can not deserialize OfferBilling: " + e.getMessage)
              }
            } ~ delete {
              receiver.delete(bizId) match {
                case Some(offer) => complete(StatusCodes.OK)
                case None => complete(StatusCodes.NotFound)
              }
            }
          case Failure(e) =>
            complete(StatusCodes.BadRequest, "Invalid id for offer billing: " + e.getMessage)
        }
      } ~ path("run" / Segment) { name =>
        schedulerApi.offer(name) match {
          case Success(result) =>
            complete(StatusCodes.OK, s"Starting task $name: $result")
          case Failure(e) =>
            complete(StatusCodes.InternalServerError, s"Failed to offer task $name: ${e.getMessage}")
        }
      } ~ pathPrefix("push" / "event" / Segment) { name =>
        (decompressRequest & (post | put) & entity(as[Array[Byte]])) { body =>
          val processed = Future.successful(0); //  hydraSkinApi.push(name, body)

          onComplete(processed) {
            case Success(count) => {
              complete(StatusCodes.OK, s"Parsed $count events")
            }
            case Failure(t) => {
              completeErrorWithMessage(s"Failed to process events. Message: ${t.getMessage}", t)
            }
          }
        }
      } ~ pathPrefix("agencies") {
        pathPrefix("operators") {
          (path("update") & longArray("id") & intArray("operator")) { (ids, operators) =>
            val opSet = operators.toSet
            for (id <- ids) {
              val notPartnered = tourOperators.getAll.map(_.id).filterNot(opSet.contains)
              agencyPartneringReceiver.receive(id, notPartnered.toSet)
            }
            completeJsonOk(new JSONObject())
          } ~ (pathEndOrSingleSlash & longArray("id")) { ids =>
            val ar = new JSONArray()
            for (id <- ids) {
              val notPartnered = agencyPartneringReceiver.get(id).getOrElse(Set.empty)
              val operatorsArray = new JSONArray()
              tourOperators.getAll.filterNot(op => notPartnered.contains(op.id)).foreach { partnered =>
                operatorsArray.put(new JSONObject().put("id", partnered.id).put("name", partnered.name))
              }
              ar.put(new JSONObject().put("id", id).put("operators", operatorsArray))
            }
            completeJsonOk(ar)
          }
        } ~ {
          http => agencyHandler ! http
        }
      }
    }
}
