package ru.yandex.tours.indexer

import org.joda.time.LocalDate
import ru.yandex.tours.geo.base.region.{Id, Tree}
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.Languages
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.search.HotelSearchRequest
import ru.yandex.tours.model.search.SearchProducts.{HotelSnippet, Offer}
import ru.yandex.tours.util.lang.Dates._

object SearchIndexUtil {
  private val validNights = 5 until 20

  def regionChain(hotel: Hotel, tree: Tree, geoMapping: GeoMappingHolder): Seq[Id] = {
    for {
      region <- tree.region(hotel.geoId).toSeq
      r <- tree.pathToRoot(region)
      if geoMapping.isKnownDestination(r.id)
    } yield r.id
  }

  def shouldIndex(request: HotelSearchRequest): Boolean = {
    if (request.when.isBefore(LocalDate.now)) return false
    if (request.kids > 0) return false
    if (request.adults != 2) return false
    if (request.currency != Currency.RUB) return false
    if (request.lang != Languages.ru) return false
    true
  }

  def shouldIndex(offer: Offer): Boolean = {
    if (!offer.getWithFlight) return false
    if (!validNights.contains(offer.getNights)) return false
    if (offer.getDate.toLocalDate.isBefore(LocalDate.now)) return false
    true
  }

  def shouldIndex(snippet: HotelSnippet): Boolean = {
    if (!snippet.getWithFlight) return false
    if (snippet.getNightsMin != snippet.getNightsMax) return false
    if (snippet.getDateMin != snippet.getDateMax) return false
    if (!validNights.contains(snippet.getNightsMin)) return false
    if (snippet.getDateMin.toLocalDate.isBefore(LocalDate.now)) return false
    if (snippet.getSourceCount != 1) return false
    true
  }
}
