package ru.yandex.tours.indexer.agencies

import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.indexer.backa.BackaExportHolder
import ru.yandex.tours.indexer.task.{PeriodicUpdatable, TaskWeight}
import ru.yandex.tours.metro.MetroHolder
import ru.yandex.tours.util.IO

import scala.concurrent.ExecutionContext
import scala.concurrent.duration.FiniteDuration

class AgenciesIndexer(backaHolder: BackaExportHolder,
                      tree: Tree,
                      dataPersistenceManager: DataPersistenceManager,
                      updateInterval: FiniteDuration,
                      extDataService: ExtDataService,
                      metroHolder: MetroHolder)
                     (implicit ec: ExecutionContext) extends PeriodicUpdatable(updateInterval, "agencies")
  with TaskWeight.Medium {

  override def run(): Unit = {
    val companies = backaHolder.getByName(backaHolder.COMPANIES)
    if (companies.isEmpty) {
      log.warn("Backa Companies file not present")
      return
    }

    val shows = getAgencyShows
    val phones = getPhoneOverrides

    val parser = new XmlAgencyParser(tree, shows, phones, metroHolder)

    val agencies = IO.using(companies.get) { backaStream ⇒
      parser.parseInputStream(backaStream)
    }
    require(agencies.nonEmpty, "Agencies index should not be empty")

    val is = IO.writeStream(os ⇒ agencies.foreach(_.writeDelimitedTo(os)))
    dataPersistenceManager.checkAndStore(DataTypes.agenciesProto, is)
  }

  private def getPhoneOverrides: Map[Long, String] = {
    SubstitutionPhonesParser.parse(extDataService.readData(DataTypes.substitutePhones))
  }

  private def getAgencyShows: Map[Long, Int] = {
    val shows = AgencyShowsParser.parse(extDataService.readData(DataTypes.agenciesShows))
    require(shows.nonEmpty, "shows should not be empty")
    shows
  }
}
