package ru.yandex.tours.indexer.agencies

import javax.xml.stream.XMLStreamReader

import org.joda.time.LocalTime
import ru.yandex.tours.geo
import ru.yandex.tours.geo.base.region.Tree
import ru.yandex.tours.metro.{MetroHolder, MetroUtil}
import ru.yandex.tours.model.Agencies.{Day2Interval, ProtoAgency, ProtoInterval, ProtoTimetable}
import ru.yandex.tours.model.Agency
import ru.yandex.tours.model.Languages.Lang
import ru.yandex.tours.model.util.proto
import ru.yandex.tours.parsing.AbstractBackaXmlParser

import scala.collection.JavaConversions._
import scala.collection.JavaConverters._
import scala.collection.mutable

class XmlAgencyParser(tree: Tree,
                      shows: Map[Long, Int],
                      phoneOverrides: Map[Long, String],
                      metroHolder: MetroHolder) extends AbstractBackaXmlParser[ProtoAgency] {

  private val AgencyRubrics = Set("184106432", "184106430")

  override protected def parseCompany(parser: XMLStreamReader) = {
    val agency = ProtoAgency.newBuilder()
    var rubrics = Set.empty[String]
    var timetable = Map.empty[Int, Seq[ProtoInterval]]
    var address = Map.empty[Lang, String]
    var ignore = false
    agency.setId(getAttribute(parser, "id").toLong)
    parseTag(parser, "company") {
      case "name" ⇒
        val lang = getLangAttribute(parser)
        val name = parseString(parser)
        agency.addName(proto.toLangVal(lang.toString, name))
      case "url" ⇒ agency.setUrl(parseString(parser))
      case "closed" ⇒ ignore = true
      case "unreliable" ⇒ ignore = true
      case "geo" ⇒
        parseTag(parser, "geo") {
          case "addressline" ⇒
            val lang = getLangAttribute(parser)
            if (!address.contains(lang)) {
              address += lang → parseString(parser)
            }
          case "address" ⇒
            parseTag(parser, "address") {
              case "formatted" ⇒
                val lang = getLangAttribute(parser)
                if (!address.contains(lang)) {
                  address += lang → parseString(parser)
                }
            }
          case "geoid" ⇒
            agency.setGeoId(parseString(parser).toInt)
          case "pos" ⇒
            val coords = parseString(parser).trim.split(" ").map(_.toDouble)
            agency.getPointBuilder
              .setLongitude(coords(0))
              .setLatitude(coords(1))
        }
      case "pos" ⇒
        val coords = parseString(parser).trim.split(" ").map(_.toDouble)
        agency.getPointBuilder
          .setLongitude(coords(0))
          .setLatitude(coords(1))
      case "phones" ⇒
        parsePhones(parser).foreach(agency.addPhones)
      case "geoid" ⇒
        agency.setGeoId(parseString(parser).toInt)
      case "rubric" ⇒
        rubrics += getAttribute(parser, "ref")
      case "hours" ⇒
        timetable = parseTimeTable(parser)
    }
    if ((rubrics & AgencyRubrics).nonEmpty && !ignore) {
      if (agency.hasGeoId) {
        tree.getTimeZone(agency.getGeoId).map { tz ⇒
          val day2intervals = timetable.map { case (day, intervals) ⇒
            Day2Interval.newBuilder().setDay(day).addAllIntervals(intervals).build
          }
          ProtoTimetable.newBuilder()
            .setTimezone(tz.getID)
            .addAllDay2Intervals(day2intervals.asJava)
        }.foreach {
          t ⇒ if (t.getDay2IntervalsCount > 0) agency.setTimetable(t)
        }
        if (agency.hasPoint) {
          val metros = for {
            region ← tree.pathToRoot(agency.getGeoId)
            metro ← metroHolder.byGeoId(region.id)
          } yield {
            metro → geo.distanceInKm(agency.getPoint, metro.getPoint)
          }
          if (metros.nonEmpty) {
            val best = metros.minBy(_._2)._1
            metroHolder.getConnected(best.getId).foreach {
              metro ⇒ agency.addMetroId(MetroUtil.compositeId(metro))
            }
          }
        }
        agency.addAllAddress(asJavaIterable(toLangVal(address)))
      }
      phoneOverrides.get(agency.getId).foreach(phone ⇒ agency.clearPhones().addPhones(phone))
      agency.setShows(shows.getOrElse(agency.getId, 0))
      Some(agency.build)
    } else {
      None
    }
  }

  private def parseTimeTable(parser: XMLStreamReader): Map[Int, Seq[ProtoInterval]] = {
    var map = Map.empty[Int, Seq[ProtoInterval]]
    parseTag(parser, "hours") {
      case "availability" ⇒ map = map ++ parseAvailability(parser)
    }
    map
  }

  private def parseTime(s: String) = {
    if (s == "24:00") {
      Agency.endOfDay
    } else {
      LocalTime.parse(s)
    }
  }

  private def parseAvailability(parser: XMLStreamReader): Map[Int, Seq[ProtoInterval]] = {
    var result = Map.empty[Int, Seq[ProtoInterval]]
    val intervals = mutable.Buffer.empty[ProtoInterval]
    val days = mutable.Buffer.empty[Int]
    parseTag(parser, "availability") {
      case "interval" ⇒
        val from = parseTime(getAttribute(parser, "from"))
        val to = parseTime(getAttribute(parser, "to"))
        val interval = ProtoInterval.newBuilder()
          .setFrom(from.getMillisOfDay)
          .setTo(to.getMillisOfDay)
          .build()
        intervals.append(interval)
      case "twentyfourhours" ⇒
        val interval = ProtoInterval.newBuilder()
          .setFrom(Agency.twentyForHourInterval.from.getMillisOfDay)
          .setTo(Agency.twentyForHourInterval.to.getMillisOfDay)
          .build()
        intervals.append(interval)
      case "monday" ⇒ days.append(1)
      case "tuesday" ⇒ days.append(2)
      case "wednesday" ⇒ days.append(3)
      case "thursday" ⇒ days.append(4)
      case "friday" ⇒ days.append(5)
      case "saturday" ⇒ days.append(6)
      case "sunday" ⇒ days.append(7)
      case "weekdays" ⇒ days.append(1, 2, 3, 4, 5)
      case "weekend" ⇒ days.append(6, 7)
      case "everyday" ⇒ days.append(1, 2, 3, 4, 5, 6, 7)
      case availability ⇒ log.warn("Unknown availability: " + availability)
    }
    if (intervals.nonEmpty) {
      for (day ← days) {
        result = result.updated(day, intervals.sortBy(_.getFrom))
      }
    }
    result
  }
}
