package ru.yandex.tours.indexer.airports

import java.io.ByteArrayInputStream

import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.avia.AviaClient
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.model.Languages
import ru.yandex.tours.util.Collections._
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.parsing.{AtSKV, Tabbed}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

class AirportsIndexer(aviaClient: AviaClient,
                      tree: region.Tree,
                      dataPersistenceManager: DataPersistenceManager,
                      updateTime: FiniteDuration)
                     (implicit ec: ExecutionContext)
  extends AsyncUpdatable(updateTime, "avia_airports") with TaskWeight.Light {

  private val ru = Languages.ru.toString

  override protected def update: Future[_] = {
    for (dictionaries <- aviaClient.getDictionaries) yield {
      val citiesByKey = dictionaries.cities.map(c => c.key -> c).toMap
      val citiesByIata = dictionaries.cities.flatMap(c => c.iata.map(_ -> c)).toMultiMap
      val countriesByKey = dictionaries.countries.map(c => c.key -> c).toMap
      val regions = (for {
        region <- tree.regions
        country <- tree.country(region).toSeq
        if region != country
        name <- region.allNames
      } yield {
        (country.id, name.toLowerCase) -> region
      }).toMap

      def geoIdByGeobase(city: AviaClient.City) = {
        for {
          country <- countriesByKey.get(city.country_key)
          countryGeoId <- country.geoid
          region <- regions.get(countryGeoId -> city.title.toLowerCase)
        } yield region.id
      }

      val airports = IO.printBytes { pw =>
        for (airport <- dictionaries.airports) {
          val cityByKey = airport.city_key.flatMap(citiesByKey.get).filter(_.geoid.isDefined)
          val cityByIata = airport.iata.flatMap(citiesByIata.get).flatMap(_.find(_.geoid.isDefined))
          val city = cityByKey.orElse(cityByIata)

          val airportName = Seq(ru -> airport.title)
          val cityName = city.map(_.title).map(ru -> _)
          val geoId = city.flatMap(_.geoid).orElse { city.flatMap(geoIdByGeobase) }
          val tz = geoId.flatMap(tree.getTimeZone)

          pw.println(Tabbed(
            AtSKV(airportName),
            AtSKV(cityName),
            geoId.fold("")(_.toString),
            city.map(_.key).getOrElse(""),
            airport.key,
            airport.iata.getOrElse(""),
            tz.fold("")(_.getID)
          ))
        }
      }

      val cities = IO.printBytes { pw =>
        for (city <- dictionaries.cities) {
          val geoid = city.geoid.orElse { geoIdByGeobase(city) }
          val tz = geoid.flatMap(tree.getTimeZone)
          pw.println(Tabbed(
            city.key,
            AtSKV(Seq(ru -> city.title)),
            geoid.fold("")(_.toString),
            city.iata.getOrElse(""),
            tz.fold("")(_.getID)
          ))
        }
      }

      dataPersistenceManager.checkAndStore(DataTypes.aviaAirports, new ByteArrayInputStream(airports))
      dataPersistenceManager.checkAndStore(DataTypes.aviaCities, new ByteArrayInputStream(cities))
    }
  }
}
