package ru.yandex.tours.indexer.backa

import java.io.{File, FileInputStream, FileOutputStream, InputStream}
import java.nio.file.StandardCopyOption._
import java.nio.file.{Files, Paths}
import java.util.zip.GZIPInputStream

import akka.util.Timeout
import com.typesafe.config.Config
import org.apache.commons.io.IOUtils
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.util.file.RichFile
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.RichConfig
import ru.yandex.tours.util.{IO, Tar}
import spray.http.StatusCodes

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration._

/*
  TODO:
    actually this should not be scheduler's task.
    it should run periodically on every node
*/
class BackaExportHolder(downloadUrl: String,
                        folder: File,
                        client: AsyncHttpClient,
                        updateTime: FiniteDuration)
                       (implicit ec: ExecutionContext)
  extends AsyncUpdatable(updateTime, "backa export") with TaskWeight.Light {

  require(folder.exists() || folder.mkdirs(), s"Working directory $folder do not exists and connot be created")

  def this(config: Config, folder: File, client: AsyncHttpClient)(implicit ec: ExecutionContext) =
    this(config.getString("tours.backa.export.url"),
      folder,
      client,
      config.getFiniteDuration("tours.backa.export.update")
    )

  val COMPANIES = "companies2.xml.gz"
  val REFERENCES = "references2.xml.gz"

  private implicit val timeout = new Timeout(10.minutes)

  private val ALL_NAMES = Set(COMPANIES, REFERENCES)

  private def fileByName(name: String) = folder / name

  def update: Future[Unit] = {
    val tmpFile = IO.newTempFile(System.currentTimeMillis().toString, "-backa-export.tar")
    val future = client.downloadToFile(downloadUrl, tmpFile).map {
      case StatusCodes.OK =>
        try {
          IO.using(new FileInputStream(tmpFile))(process)
          log.info("Backa export successfully updated")
        } catch {
          case e: Exception => log.warn("Can not update backa export", e)
        }
      case rc => log.warn(s"Backa export is not updated. Response code is $rc")
    }
    future andThen { case _ =>
      IO.deleteFile(tmpFile)
    }
  }

  private def process(is: InputStream): Unit = {
    Tar.iterate(is) { (name, fileStream) =>
      if (ALL_NAMES.contains(name)) {
        log.info(s"Backa export: processing file $name")
        val output = new File(folder, System.currentTimeMillis() + name)
        IO.using(new FileOutputStream(output)) { os =>
          IOUtils.copyLarge(fileStream, os)
        }
        val from = Paths.get(output.getAbsolutePath)
        val to = Paths.get(fileByName(name).getAbsolutePath)
        Files.move(from, to, ATOMIC_MOVE, REPLACE_EXISTING)
      } else {
        log.info(s"Backa export: skip file $name")
      }
    }
  }

  def getByName(name: String): Option[InputStream] = {
    val file = fileByName(name)
    if (file.exists()) {
      Some(new GZIPInputStream(new FileInputStream(file)))
    } else {
      None
    }
  }

  def getUpdateTime() : FiniteDuration = updateTime
}
