package ru.yandex.tours.indexer.backa

import java.io.InputStream

import ru.yandex.common.xml.{XParser, XTag}
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.indexer.task.{PeriodicUpdatable, TaskWeight}
import ru.yandex.tours.util.IO
import java.io.{FileInputStream, PrintWriter}

import ru.yandex.tours.util.parsing.Tabbed

import scala.concurrent.ExecutionContext
import scala.concurrent.duration.FiniteDuration

class BackaPermalinkIndexer(backaHolder: BackaExportHolder,
                      dataPersistenceManager: DataPersistenceManager,
                      updateInterval: FiniteDuration,
                      extDataService: ExtDataService)
                     (implicit ec: ExecutionContext) extends PeriodicUpdatable(updateInterval, "backapermalinks_index")
  with TaskWeight.Light {

  override def run(): Unit = {
    val referencesXml = backaHolder.getByName(backaHolder.REFERENCES)
    if (referencesXml.isEmpty) {
      log.warn("Backa References file not present")
      return
    }

    val file = IO.newTempFile("backapermalinks")
    IO.using(new PrintWriter(file)) { pw =>
      IO.using(referencesXml.get) { backaStream => parseAndWrite(backaStream, pw) }
    }
    dataPersistenceManager.checkAndStore(DataTypes.backaPermalinks, new FileInputStream(file))
    file.delete()
  }

  private def parseAndWrite(is: InputStream, pw: PrintWriter) : Unit = {
    var cnt = 0
    new XParser {
      override def build(): XTag = new XTag() {
        tag("Company") = handle {
          val backaId = attr("id")
          var hotelId : Int = 0
          tag("reference") = handleComplete {
            if (attr("scope") == "yandex_travel") {
              hotelId = text.toInt
            }
          }
          complete {
            if (hotelId != 0) {
              cnt += 1
              pw.println(Tabbed(hotelId, backaId))
            }
          }
        }
      }
    }.parse(is)
    log.info(s"Backa permalinks indexed, have $cnt lines")
  }
}
