package ru.yandex.tours.indexer.billing

import java.io.{ByteArrayInputStream, InputStream}

import ru.yandex.extdata.common.exception.{DataIsNotLoadedException, UnableToLoadDataException}
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.indexer.task.PeriodicUpdatable

import scala.collection.concurrent.TrieMap
import scala.concurrent.ExecutionContext
import scala.concurrent.duration.FiniteDuration
import scala.util.Try

abstract class AbstractInfoReceiver[T](extDataService: ExtDataService,
                                     dataPersistenceManager: DataPersistenceManager,
                                     updateTime: FiniteDuration,
                                     dataType: DataType)
                                    (implicit ex: ExecutionContext)
  extends PeriodicUpdatable(updateTime, dataType.getName) {

  protected def parseIs(is: InputStream): Iterable[(Long, T)]
  
  protected def dump(info: Iterable[(Long, T)]): Array[Byte]
  
  private val id2offer = {
    val is = try {
      extDataService.readData(dataType)
    } catch {
      case _: DataIsNotLoadedException | _: UnableToLoadDataException =>
        log.warn(s"No previous ${dataType.getName}. Using empty one.")
        new ByteArrayInputStream(Array.empty[Byte])
    }
    val offers = Try(parseIs(is)).getOrElse {
      log.warn(s"Can not read previous ${dataType.getName}!! Using empty one")
      Iterable.empty[(Long, T)]
    }
    TrieMap(offers.toSeq: _*)
  }

  def receive(id: Long, info: T): Unit = {
    id2offer.put(id, info)
  }

  def delete(id: Long) = {
    id2offer.remove(id)
  }

  def get(id: Long) = {
    id2offer.get(id)
  }

  override def run(): Unit = {
    val array = dump(id2offer)
    dataPersistenceManager.checkAndStore(dataType, new ByteArrayInputStream(array))
  }
}
