package ru.yandex.tours.indexer.clusterization

import java.io.{Closeable, File, FileOutputStream}
import java.util.concurrent.Callable

import com.google.common.cache.{CacheBuilder, RemovalListener, RemovalNotification}
import com.google.protobuf.{AbstractMessageLite, Parser}
import org.apache.commons.io.FileUtils
import ru.yandex.tours.db.GridPoint
import ru.yandex.tours.geo
import ru.yandex.tours.indexer.clusterization.Grid._
import ru.yandex.tours.model.BaseModel.Point
import ru.yandex.tours.util.{IO, ProtoIO}

import scala.collection.immutable.IndexedSeq

/* @author berkut@yandex-team.ru */

object Grid {
  private[clusterization] val MIN_LATITUDE = -90
  private[clusterization] val MAX_LATITUDE = 90
  private[clusterization] val MIN_LONGITUDE = -180
  private[clusterization] val MAX_LONGITUDE = 180
  private[clusterization] val MULTIPLIER = 20
  private[clusterization] val LON_SIZE = (Grid.MAX_LONGITUDE - Grid.MIN_LONGITUDE) * Grid.MULTIPLIER
  private[clusterization] val LAT_SIZE = (Grid.MAX_LATITUDE - Grid.MIN_LATITUDE) * Grid.MULTIPLIER

  def readFromFolder[T](folderPath: String, parser: Parser[T]): GridInFile[T] = {
    val folder = new File(folderPath)
    if (!folder.exists() || !folder.isDirectory) sys.error(s"$folderPath doesn't exists or is not folder")
    new GridInFile[T](folder, parser)
  }

  def dumpToFolder[T <: AbstractMessageLite[_, _]](folderPath: String,
                                             hotels: TraversableOnce[T],
                                             getPoint: T => Point,
                                             overWrite: Boolean = true): Unit = {
    val rootFolder = new File(folderPath)
    if (rootFolder.exists()) {
      if (overWrite) {
        FileUtils.deleteDirectory(rootFolder)
      } else {
        sys.error("Folder for grid already exists!")
      }
    }
    if (!rootFolder.mkdirs()) {
      sys.error(s"Could not create folders: $folderPath")
    }

    val osCache = CacheBuilder.newBuilder()
      .maximumSize(50)
      .removalListener(new RemovalListener[(Int, Int), FileOutputStream] {
        override def onRemoval(notification: RemovalNotification[(Int, Int), FileOutputStream]): Unit = {
          notification.getValue.close()
        }
      })
      .build[(Int, Int), FileOutputStream]()

    for (hotel <- hotels) {
      val (lonIdx, latIdx) = getIndex(getPoint(hotel))

      val os = osCache.get((lonIdx, latIdx), new Callable[FileOutputStream] {
        override def call(): FileOutputStream = {
          val folder = new File(s"$folderPath/$lonIdx")
          FileUtils.forceMkdir(folder)
          val output = new File(folder, s"$latIdx")
          new FileOutputStream(output, true)
        }
      })

      hotel.writeDelimitedTo(os)
    }

    osCache.invalidateAll()
  }

  def getIndex(point: Point): (Int, Int) = {
    getIndex(point.getLongitude, point.getLatitude)
  }

  def getIndex(lon: Double, lat: Double): (Int, Int) = {
    if (lat < MIN_LATITUDE || lat > MAX_LATITUDE) {
      sys.error("wrong latitude value:" + lat)
    }
    if (lon < MIN_LONGITUDE || lon > MAX_LONGITUDE) {
      sys.error("wrong longitude value: " + lon)
    }
    ((lon - MIN_LONGITUDE) * MULTIPLIER).toInt → ((lat - MIN_LATITUDE) * MULTIPLIER).toInt
  }

  def getGeoPoint(point: Point): GridPoint = {
    GridPoint.fromPoint(point)
  }

  def fromIndex(lonIdx: Int, latIdx: Int): (Double, Double) = {
    (lonIdx.toDouble / MULTIPLIER + MIN_LONGITUDE) → (latIdx.toDouble / MULTIPLIER + MIN_LATITUDE)
  }
}

trait Grid[T] extends Closeable {

  def get(lonIdx: Int, latIdx: Int): Iterator[T]

  def getNear(lon: Double, lat: Double): Iterator[T] = {
    val (lonIndex, latIndex) = getIndex(lon, lat)
    for {
      lon <- (lonIndex - 1 to lonIndex + 1).iterator
      lat <- (latIndex - 1 to latIndex + 1).iterator
      hotel <- get(lon, lat)
    } yield hotel
  }

  def getNear(point: Point): Iterator[T] = {
    getNear(point.getLongitude, point.getLatitude)
  }

  def iterator: Iterator[T] = {
    val indexes = for {
      lonIdx <- 0 until Grid.LON_SIZE
      latIdx <- 0 until Grid.LAT_SIZE
    } yield (lonIdx, latIdx)
    indexes.toIterator.flatMap { case (i, j) => get(i, j) }
  }

  private lazy val summary = {
    var total = 0
    var maxInCell = 0
    var cells = 0
    for {
      lonIdx <- 0 until Grid.LON_SIZE
      latIdx <- 0 until Grid.LAT_SIZE
    } {
      val cellSize = get(lonIdx, latIdx).size
      if (cellSize > 0) {
        total += cellSize
        maxInCell = Math.max(maxInCell, cellSize)
        cells += 1
      }
    }
    (total, maxInCell, cells)
  }

  def size: Int = summary._1

  def getGridSummary: String = {
    val (total, maxInCell, cells) = summary
    s"total: $total, maxInCell: $maxInCell, cells: $cells, avgInCell: ${total.toDouble / cells}"
  }
}

class GridInFile[T](folder: File, parser: Parser[T]) extends Grid[T] {
  require(folder.exists())

  private def getPath(long: Int, lat: Int) = new File(folder, s"$long/$lat")

  override def get(lonIdx: Int, latIdx: Int): Iterator[T] = {
    val normalizedLon = lonIdx % Grid.LON_SIZE
    val normalizedLat = latIdx % Grid.LAT_SIZE
    val path = getPath(normalizedLon, normalizedLat)
    if (!path.exists()) {
      Iterator.empty
    } else {
      ProtoIO.loadFromFile(path, parser)
    }
  }

  override def close(): Unit = IO.deleteFile(folder)

  override def toString: String = s"GridInFile($folder, $getGridSummary)"
}

class GridInMemory[T](objs: Seq[T], getPoint: T => Point) extends Grid[T] {
  override def get(lonIdx: Int, latIdx: Int): Iterator[T] = {
    objs.filter { t =>
      val p = getPoint(t)
      getIndex(p) == (lonIdx, latIdx)
    }.iterator
  }

  override def close(): Unit = {}
}
