package ru.yandex.tours.indexer.clusterization

import akka.actor.ActorSystem
import ru.yandex.tours.clustering.Clustering.LinkWithConfidence
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.dao.HotelsDao.{SkipDeleted, WithIds}
import ru.yandex.tours.db.tables.Clusterization
import ru.yandex.tours.db.{DBWrapper, Transaction}
import ru.yandex.tours.hotels.clustering.{ClusteringModel, LocalContext}
import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.collections.RafBasedMap

import scala.collection.JavaConverters._
import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
  * Created by asoboll on 09.03.17.
  */
abstract class MappingValidationClusterizer(override val name: String,
                                            dbWrapper: DBWrapper,
                                            hotelsDao: HotelsDao,
                                            clusteringModel: ClusteringModel)
                                           (implicit akka: ActorSystem, ec: ExecutionContext)
  extends NewHotelClusterizer(dbWrapper, hotelsDao, clusteringModel, 1.hour) {

  protected val mappingDescription: Seq[(Partner, String)] // target partner and param with mapping

  private def suggestedPartnerInfos(hotel: HotelRef,
                                    hotelsMap: collection.Map[Int, PartnerHotel]): Seq[(Int, String)] = {
    for {
      partnerHotel <- hotelsMap.get(hotel.id).toSeq
      infoList = partnerHotel.getRawHotel.getAddInfoList.asScala
      (partnerToMap, paramName) <- mappingDescription
      addInfo <- infoList.find(_.getName == paramName)
    } yield (partnerToMap.id, addInfo.getValue)
  }

  private def getPartnerInfo(hotel: HotelRef,
                             hotelsMap: collection.Map[Int, PartnerHotel]): Option[(Int, String)] = {
    hotelsMap.get(hotel.id).map { hotel =>
      (hotel.getRawHotel.getPartner, hotel.getRawHotelOrBuilder.getPartnerId)
    }
  }

  override protected def doClustering(hotelsToCluster: Seq[HotelRef],
                                      candidates: Seq[HotelRef],
                                      cleaner: HotelLinkCleaner,
                                      metrics: ClusteringMetrics,
                                      transaction: Transaction) = {

    val map = getHotelMap(candidates.map(_.id).toSet ++ hotelsToCluster.map(_.id).toSet)

    val links: Future[Seq[LinkWithConfidence]] = {
      for (hotelsMap ← map) yield {
        val localContext = new LocalContext(hotelsMap.valuesIterator)

        val suggestedMap = (for {
          ref <- hotelsToCluster
          suggested = suggestedPartnerInfos(ref, hotelsMap)
          if suggested.nonEmpty
        } yield ref -> suggested).toMap
        val byPartnerInfoMap = (for {
          candidate <- candidates
          partnerInfo <- getPartnerInfo(candidate, hotelsMap)
        } yield partnerInfo -> candidate).toMap

        (for {
          (hotelToCluster, partnerInfos) ← suggestedMap
          hotelsToCluster = Seq(hotelToCluster)
          contexts = buildHotelContexts(hotelsToCluster, localContext, hotelsMap)
          groupedCandidates = partnerInfos.flatMap(byPartnerInfoMap.get)
          candidateContexts = buildHotelContexts(groupedCandidates, localContext, hotelsMap)
          link ← generateLinks(hotelsToCluster, groupedCandidates, candidateContexts ++ contexts, metrics).seq
        } yield link).toVector
      }
    }
    links.onComplete { _ ⇒
      map.foreach {
        case m: RafBasedMap[_, _] ⇒ m.close()
        case _ ⇒
      }
    }

    for {
      links ← links
      cleaned = cleaner.removeExcessLinksAndSetConfidence(links)
      _ ← saveLinksAndPublish(hotelsToCluster, cleaned, transaction, metrics)
    } yield metrics.doLogging()
  }

  override protected def shouldPublish(id: Int, linkConf: Map[Int, List[Double]]): Boolean = {
    linkConf.getOrElse(id, Seq.empty)
      .exists(_ >= Clusterization.defaultMinConfidence)
  }

}
