package ru.yandex.tours.indexer.currency

import akka.util.Timeout
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.model.BaseModel.Currency
import ru.yandex.tours.model.BaseModel.Currency.RUB
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.http.AsyncHttpClient
import ru.yandex.tours.util.lang.Futures
import ru.yandex.tours.util.parsing.Tabbed
import spray.http.{StatusCodes, Uri}

import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, Future}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 06.04.16
 */
class CurrencyRateDownloader(httpClient: AsyncHttpClient,
                             dataPersistenceManager: DataPersistenceManager,
                             updateTime: FiniteDuration)
                            (implicit ec: ExecutionContext)
  extends AsyncUpdatable(updateTime, "currency_rates") with TaskWeight.Light {

  private implicit val timeout = Timeout(1.minute)

  def currenciesToLoad: Array[Currency] = Currency.values().filter(_ != RUB)

  def parseResponse(line: String): BigDecimal = BigDecimal(line.split(",").apply(1))
  def uri(from: Currency, to: Currency): Uri = {
    Uri(s"http://download.finance.yahoo.com/d/quotes.csv").withQuery(
      "e" -> ".csv",
      "f" -> "sl1d1t1",
      "s" -> s"$from$to=X"
    )
  }

  override protected def update: Future[_] = {
    val futures = currenciesToLoad.iterator.map { currency =>
      for {
        (sc1, toRubRate) <- httpClient.get(uri(from = currency, to = RUB))
        (sc2, fromRubRate) <- httpClient.get(uri(from = RUB, to = currency))
      } yield {
        require(sc1 == StatusCodes.OK, s"Unexpected response from yahoo: sc = $sc1, body = $toRubRate")
        require(sc2 == StatusCodes.OK, s"Unexpected response from yahoo: sc = $sc2, body = $fromRubRate")
        Map(
          (currency -> RUB) -> parseResponse(toRubRate),
          (RUB -> currency) -> parseResponse(fromRubRate)
        )
      }
    }
    val combined = Futures.lazyFold(futures, Map.empty[(Currency, Currency), BigDecimal])(_ ++ _)

    combined.map { map =>
      val is = IO.printStream { pw =>
        for (((from, to), rate) <- map) {
          pw.println(Tabbed(from, to, rate))
        }
      }
      dataPersistenceManager.checkAndStore(DataTypes.currencyRates, is)
    }
  }
}
