package ru.yandex.tours.indexer.data

import java.io.File
import java.math.RoundingMode

import akka.stream.Attributes
import akka.stream.scaladsl.Flow
import akka.stream.stage._
import com.google.common.math.IntMath
import ru.yandex.tours.util.collections.DumpingQueue
import ru.yandex.vertis.curator.recipes.map.ValueSerializer

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 11.01.16
 */
class FileBuffer[T](dir: File,
                    valueSerializer: ValueSerializer[T],
                    maxMemorySize: Int,
                    maxSize: Int) extends DetachedStage[T, T] {

  val buffer = DumpingQueue(dir, valueSerializer, maxMemorySize)

  override def onPush(elem: T, ctx: DetachedContext[T]): UpstreamDirective = {
    if (ctx.isHoldingDownstream) {
      val el = buffer.pull()
      if (el.isEmpty) ctx.pushAndPull(elem)
      else {
        buffer.push(elem)
        ctx.pushAndPull(elem)
      }
    } else {
      buffer.push(elem)
      if (buffer.size >= maxSize) ctx.holdUpstream()
      else ctx.pull()
    }
  }

  override def onPull(ctx: DetachedContext[T]): DownstreamDirective = {
    if (ctx.isFinishing) {
      val elem = buffer.pull()
      if (elem.nonEmpty) {
        if (buffer.isEmpty) ctx.pushAndFinish(elem.get)
        else ctx.push(elem.get)
      } else {
        ctx.finish()
      }
    } else if (ctx.isHoldingUpstream) {
      val elem = buffer.pull()
      if (elem.isEmpty) sys.error("unexpected")
      else ctx.pushAndPull(elem.get)
    }
    else if (buffer.isEmpty) ctx.holdDownstream()
    else {
      val elem = buffer.pull()
      if (elem.isEmpty) ctx.holdDownstream()
      else ctx.push(elem.get)
    }
  }


  override def onUpstreamFinish(ctx: DetachedContext[T]): TerminationDirective =
    if (buffer.isEmpty) ctx.finish()
    else ctx.absorbTermination()
}

object FileBuffer {
  def apply[T](dir: File,
               valueSerializer: ValueSerializer[T],
               maxMemorySize: Int = 1000,
               maxSize: Int = 1000000): Flow[T, T, Unit] = {
    val inputBuffer = math.max(1 << IntMath.log2(maxMemorySize, RoundingMode.DOWN), 16)
    Flow[T].transform(() => new FileBuffer(dir, valueSerializer, maxMemorySize, maxSize))
      .addAttributes(Attributes.inputBuffer(inputBuffer, inputBuffer))
  }
}