package ru.yandex.tours.indexer.data

import java.io.ByteArrayInputStream

import org.joda.time.DateTime
import ru.yandex.tours.events.SearchEvent
import ru.yandex.tours.util.lang.FutureTry
import ru.yandex.tours.util.{Logging, ProtoIO}
import ru.yandex.vertis.hydra.model.msg.Schema

import scala.collection.JavaConverters._
import scala.concurrent.{ExecutionContext, Future}
import scala.util.{Failure, Success, Try}

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.12.15
 */
class HydraSinkApi(searchEventStreams: SearchEventStreams)
                  (implicit ec: ExecutionContext) extends Logging {

  def push(name: String, body: Array[Byte]): Future[Int] = {
    parseEvents(body) match {
      case Success(events) =>
        for {
          res <- searchEventStreams.push(name, events)
        } yield {
          if (!res) sys.error(s"Failed to enqueue ${events.size} events: Buffer overflow")
          events.size
        }
      case Failure(e) =>
        log.error(s"Failed to parse events: ${e}")
        Future.successful(0)
    }
  }

  private def parseEvents(body: Array[Byte]) = Try {
    ProtoIO.loadFromStream(
      new ByteArrayInputStream(body),
      Schema.EventMessage.PARSER
    ).map { raw =>
      val time = new DateTime(raw.getTimestamp)
      val map = raw.getTupleList.asScala.map { t =>
        t.getKey -> t.getValue
      }.toMap
      SearchEvent.parseEvent(time, raw.getComponent, map)
    }.toVector.filter(se => !se.isInstanceOf[SearchEvent.OstrovokResult])
  }
}
