package ru.yandex.tours.indexer.data

import java.io.File
import java.util.concurrent.ConcurrentHashMap

import akka.actor.ActorSystem
import akka.event.Logging.ErrorLevel
import akka.stream.scaladsl.{Keep, Sink, Source}
import akka.stream.{ActorMaterializer, Attributes, OverflowStrategy, SourceQueue}
import ru.yandex.tours.events.SearchEvent
import ru.yandex.tours.events.SearchEvent.OstrovokResult
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.file._
import ru.yandex.vertis.curator.recipes.map.{StringValueSerializer, ValueSerializer}

import scala.concurrent.Future
import scala.util.Try

/**
 * Author: Vladislav Dolbilov (darl@yandex-team.ru)
 * Created: 30.12.15
 */
class SearchEventStreams(akkaSystem: ActorSystem, dir: File) extends Logging {

  private implicit val actorMaterializer = ActorMaterializer()(akkaSystem)

  private val map = new ConcurrentHashMap[String, SourceQueue[Seq[SearchEvent]]]()

  def register[S](name: String, sink: Sink[Seq[SearchEvent], S]): S = {
    require(!map.contains(name), s"Already registered sink with name $name")
    val source = Source.queue[Seq[SearchEvent]](10, OverflowStrategy.backpressure)

    val fileBuffer = FileBuffer(dir / name, SearchEventsSerializer, maxMemorySize = 10, maxSize = 10000000)
    val asyncSink = sink.addAttributes(Attributes.asyncBoundary and Attributes.logLevels(onFinish = ErrorLevel))

    val (queue, res) = source
      .via(fileBuffer)
      .toMat(asyncSink)(Keep.both).run()

    map.putIfAbsent(name, queue)
      .ensuring(_ eq null, s"Sink with name $name registered twice")

    log.info(s"Registered SearchEvent sink with name $name")
    res
  }

  def push(name: String, searchEvent: Seq[SearchEvent]): Future[Boolean] = {
    val queue = map.get(name)
    require(queue ne null, s"Sink with name $name not registered")
    queue.offer(searchEvent)
  }

  object SearchEventsSerializer extends ValueSerializer[Seq[SearchEvent]] {
    override def serialize(elems: Seq[SearchEvent]): Array[Byte] = {
      StringValueSerializer.serialize(elems.map(_.toTSKV).mkString("\n"))
    }

    override def deserialize(data: Array[Byte]): Try[Seq[SearchEvent]] =
      StringValueSerializer.deserialize(data).map(_.split("\n").map(SearchEvent.parseLine)
        .filter(se => !se.isInstanceOf[OstrovokResult]))
  }
}