package ru.yandex.tours.indexer.export

import java.io.{File, FileOutputStream, PrintWriter}
import java.util.zip.GZIPOutputStream

import org.apache.commons.io.FileUtils
import ru.yandex.tours.export.BackaFormatExporter
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingHolder
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.model.hotels.Hotel
import ru.yandex.tours.model.hotels.HotelsHolder.HotelType
import ru.yandex.tours.search.settings.SearchSettingsHolder
import ru.yandex.tours.serialize.UrlBuilder
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.s3.S3Client
import ru.yandex.tours.util.file._

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}
import scala.util.control.NonFatal

class BackaExporter(hotelsIndex: HotelsIndex,
                    tree: region.Tree,
                    s3Client: S3Client,
                    urlBuilder: UrlBuilder,
                    geoMapping: GeoMappingHolder,
                    searchSettings: SearchSettingsHolder,
                    hcId2Key: Map[String, String],
                    updateTime: FiniteDuration)
                   (implicit ec: ExecutionContext)
  extends AsyncUpdatable(updateTime, "backa_export") with TaskWeight.Medium {

  private val countries = Map(
    -1  -> "full_backa_export.xml.gz",  // Full export with appartments and some special fields, not consumed by backa
    0   -> "backa_export.xml.gz",       // For all countries
    225 -> "backa_export.ru.xml.gz",
    187 -> "backa_export.uk.xml.gz",
    159 -> "backa_export.kz.xml.gz",
    149 -> "backa_export.bl.xml.gz"
  )

  private def genExports(): Map[String, File] = {
    val dir = IO.newTempDir("backa_export")
    def newExporter(name: String) = {
      new BackaFormatExporter(new PrintWriter(new GZIPOutputStream(new FileOutputStream(dir / name))),
        urlBuilder, tree, geoMapping, searchSettings, hcId2Key)
    }

    try {
      val exporters = countries.map {
        case (id, fileName) ⇒ id → newExporter(fileName)
      }

      for {
        hotel ← hotelsIndex.hotels
        _ = exporters(-1).add(hotel, extraFields = true)
        if shouldExport(hotel)
        _ = exporters(0).add(hotel)
        country ← tree.country(hotel.geoId)
        export ← exporters.get(country.id)
      } {
        export.add(hotel)
      }
      exporters.values.foreach(_.close())

      countries.map {
        case (_, fileName) ⇒ fileName → (dir / fileName)
      }
    } catch {
      case NonFatal(e) ⇒
        FileUtils.deleteDirectory(dir)
        throw e
    }
  }

  private def shouldExport(hotel: Hotel): Boolean = {
    hotel.`type` != HotelType.APARTMENTS
  }

  override protected def update: Future[_] = {
    for {
      exports ← Future { genExports() }
      uploads ← Future.traverse(exports) {
        case (fileName, file) ⇒
          log.info(s"Uploading $file as [$fileName] with size ${FileUtils.byteCountToDisplaySize(file.length())}")
          s3Client.uploadFile(fileName, file)
      }.andThen {
        case _ ⇒
          log.info(s"Uploading exports finished")
          exports.values.foreach(IO.deleteFile)
      }
    } yield ()
  }
}
