package ru.yandex.tours.indexer.export

import java.io.File

import ru.yandex.tours.indexer.task.AsyncUpdatable
import ru.yandex.tours.util.IO
import ru.yandex.tours.util.s3.S3Client
import spray.http.StatusCodes

import scala.concurrent.{ExecutionContext, Future}
import scala.concurrent.duration.FiniteDuration
import scala.util.{Failure, Success}

abstract class Exporter(name: String,
                        filename: String,
                        s3Client: S3Client,
                        updateTime: FiniteDuration)
                       (implicit ec: ExecutionContext)
  extends AsyncUpdatable(updateTime, name) {

  override protected def update: Future[_] = {
    generateExport().flatMap { file =>
      upload(file).andThen {
        case _ => IO.deleteFile(file)
      }
    }
  }

  private def upload(file: File): Future[_] = {
    s3Client.uploadFile(filename, file) andThen {
      case Success((StatusCodes.OK, _)) => log.info(s"$name successfully updated to S3")
      case Success((sc, res)) => log.warn(s"Can not upload $name to S3! Sc is $sc. Response: $res")
      case Failure(e) => log.error(s"Can not upload $name to S3", e)
    }
  }

  protected def generateExport(): Future[File]
}
