package ru.yandex.tours.indexer.geomapping

import java.io.ByteArrayInputStream

import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.avia.Airports
import ru.yandex.tours.db.DBWrapper
import ru.yandex.tours.db.geomapping._
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.region
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.util.IO
import slick.driver.MySQLDriver.api._

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

/**
  * Created by asoboll on 02.12.15.
  */
class GeoMappingIndexer(dataTypes: List[DataType],
                        db: DBWrapper,
                        regionTree: region.Tree,
                        airports: Airports,
                        dataPersistenceManager: DataPersistenceManager,
                        updateTime: FiniteDuration)
                       (implicit ex: ExecutionContext)
  extends AsyncUpdatable(updateTime, "geo_mapping_indexer") with TaskWeight.Light {

  private def toByteArray(mapping: Iterable[GeoMappingShort]) = {
    IO.printBytes { pw =>
      mapping.foreach(geoMappingShort => pw.println(geoMappingShort.toTsv))
    }
  }

  private def store(dataType: DataType, bannedSet: Set[Int]) = {
    val query = GeoMappingTables.queryAll(dataType)
    for {
      result <- db.run(query.result.transactionally)
      mapping = GeoMappingRecordProcessor.getMapping(result.map(GeoMappingRecordExt.tupled), bannedSet)
      mappingWithKnownDepartures = filterDeparturesWithAirports(dataType, mapping)
      array = toByteArray(mappingWithKnownDepartures.toSeq.sorted)
    } yield dataPersistenceManager.checkAndStore(dataType, new ByteArrayInputStream(array))
  }

  private def getBannedSet: Future[Set[Int]] = {
    val bannedQueries = dataTypes.map(dataType => db.run(GeoMappingTables.queryBanned(dataType).result))
    for {
      bannedList <- Future.sequence(bannedQueries)
      banned = for {
        records <- bannedList
        id <- GeoMappingRecordProcessor.getBanned(records)
        region <- regionTree.region(id).toSeq
        child <- regionTree.allChildren(region)
      } yield child.id
    } yield banned.toSet
  }

  override protected def update: Future[_] = {
    val futureStore = for{
      bannedSet <- getBannedSet
      storeList = for {
        dataType <- dataTypes
      } yield store(dataType, bannedSet)
    } yield Future.sequence(storeList)
    futureStore.flatMap(identity)
  }

  private def filterDeparturesWithAirports(dataType: DataType, mapping: Iterable[GeoMappingShort]) = {
    if (dataType != DataTypes.departures) mapping
    else mapping.filter { record =>
      airports.byGeoId(record.geoId).nonEmpty
    }
  }
}