package ru.yandex.tours.indexer.geomapping

import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.partners.{MappedPartnerTree, PartnerRegionHotelCounts, PartnerRegionsDef}
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.util.{IO, Logging}
import shapeless.HNil

import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}
import scala.util.Try

/**
  * Created by asoboll on 13.03.17.
  */
class PartnerTreesIndexer(extDataService: ExtDataService,
                          dataPersistenceManager: DataPersistenceManager,
                          updateTime: FiniteDuration)
                         (implicit ex: ExecutionContext)
  extends AsyncUpdatable(updateTime, "partner_trees_indexer") with TaskWeight.Light with Logging {

  override def update: Future[_] = Future {
    val partnerRegionHotelCounts = PartnerRegionHotelCounts.from(extDataService)
    val partnerTrees = PartnerRegionsDef.from(partnerRegionHotelCounts :: HNil, extDataService)

    val action = for {
      (partner, dt) <- DataTypes.partnerTrees
      tree <- partnerTrees.get(partner)
    } yield partner -> Try {
      val is = IO.writeStream(MappedPartnerTree.writeTo(_, tree))
      dataPersistenceManager.checkAndStore(dt, is)
    }

    val failed = action.filter(_._2.isFailure)
    if (failed.nonEmpty)
      log.error(s"failed to build region trees for partners: [${failed.keys.mkString(", ")}]",
        failed.valuesIterator.next.failed.get)
  }
}
