package ru.yandex.tours.indexer.geomapping

import java.io.ByteArrayInputStream

import com.typesafe.config.Config
import ru.yandex.extdata.common.meta.DataType
import ru.yandex.extdata.common.service.ExtDataService
import ru.yandex.extdata.loader.engine.DataPersistenceManager
import ru.yandex.tours.db.geomapping._
import ru.yandex.tours.db.{DBWrapper, MatchingResult, MatchingStats, Transactions}
import ru.yandex.tours.extdata.DataTypes
import ru.yandex.tours.geo.base.{GeoSynonyms, region}
import ru.yandex.tours.geo.mapping.GeoMappingShort
import ru.yandex.tours.geo.matching._
import ru.yandex.tours.geo.partners.{PartnerTree, PartnerTrees}
import ru.yandex.tours.hotels.HotelsIndex
import ru.yandex.tours.indexer.task.{AsyncUpdatable, TaskWeight}
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.{IO, Logging}
import slick.driver.MySQLDriver.api._

import scala.collection.JavaConversions._
import scala.concurrent.duration.FiniteDuration
import scala.concurrent.{ExecutionContext, Future}

/**
  * Created by asoboll on 01.02.16.
  */
class PartnersGeoMatcher(geobaseTree: region.Tree,
                         partnerTrees: PartnerTrees,
                         hotelsIndex: HotelsIndex,
                         config: Config,
                         db: DBWrapper,
                         extDataService: ExtDataService,
                         updateTime: FiniteDuration,
                         dataPersistenceManager: DataPersistenceManager)
                        (implicit ex: ExecutionContext)
  extends AsyncUpdatable(updateTime, "partners_geo_matcher") with TaskWeight.Medium with Logging {

  private val partners = config.getStringList("partners").toSeq.map(Partners.withName)
  private val maxDropPercentage = config.getDouble("max-drop-percentage")
  private val dataTypes = Iterable(DataTypes.countries, DataTypes.cities, DataTypes.departures)

  private def matchPartner(partnerTree: PartnerTree, geoMatcher: GeoMatcher,
                           oldMappings: Map[DataType, Seq[GeoMappingRecordExt]]) = {
    val mapping = oldMappings.values.flatMap(GeoMappingRecordProcessor.getMapping(_))
      .groupBy(_.partnerId).mapValues(_.map(_.geoId))
    geoMatcher.run(partnerTree, mapping)
  }

  private def createPatch(newHyps: Iterable[Hypothesis],
                          oldMapping: Seq[GeoMappingRecordExt]): Set[(Command, GeoMappingShort)] = {
    val automaticRecords = oldMapping.filterNot(_.isManual)
    val mapping = GeoMappingRecordProcessor.getMapping(automaticRecords)
    GeoMappingRecordProcessor.fullDiff(newHyps.map(_.idForm), mapping)
  }

  private def isPatchGood(patch: Set[(Command, GeoMappingShort)], hypsSize: Int): Boolean = {
    val dropCount = patch.count { case (command, _) => command == Drop }
    if (dropCount * 100 > hypsSize * maxDropPercentage) false else true
  }

  private def uploadPatch(dataType: DataType, patchData: Set[(Command, GeoMappingShort)],
                          patchIsGood: Boolean): Future[Option[Int]] =
    if (patchData.nonEmpty) {
      Transactions.withTransaction(db, isEnabled = patchIsGood) { transaction =>
        GeoMappingTables.put(db, dataType, transaction, patchData).map(_ => Some(transaction.id))
      }
    } else {
      Future.successful(None)
    }

  private def storeStats(partner: Partner, dataType: DataType, matchingResults: MatchingResults,
                         patchData: Set[(Command, GeoMappingShort)], transactionId: Int) = {
    val query = MatchingStats.table +=
      MatchingResult(
        partner,
        dataType.getName.split("[.]")(0),
        matchingResults.ok.count(_.hasType(dataType)),
        matchingResults.notResolved.count(_.hasType(dataType)),
        patchData.count { case (command, _) => command == Add },
        patchData.count { case (command, _) => command == Drop },
        transactionId)
    db.run(query)
  }

  private def storeHypotheses(hypotheses: Iterable[Hypothesis]) = {
    val dataType = DataTypes.geoMatchingHypotheses
    val array = IO.printBytes(pw => hypotheses.foreach(h => pw.println(h.toTsv)))
    dataPersistenceManager.checkAndStore(dataType, new ByteArrayInputStream(array))
  }

  def updatePartner(partner: Partner, geoMatcher: GeoMatcher,
                    oldMappingsAll: Map[DataType, Seq[GeoMappingRecordExt]]): Future[Iterable[Hypothesis]] = {
    log.info(s"matching $partner regions")
    if (!partnerTrees.contains(partner)) {
      log.warn(s"no region tree for partner: $partner")
      return Future.successful(Iterable.empty)
    }
    val partnerTree = partnerTrees(partner)
    val oldMappings = oldMappingsAll.mapValues(_.filter(_.geoMappingRecord.partner == partner))
    val matchingResults = matchPartner(partnerTree, geoMatcher, oldMappings)
    val newMappings = dataTypes.map { dataType =>
      dataType -> matchingResults.ok.filter(_.hasType(dataType))
    }.filter(_._2.nonEmpty).toMap
    Future.sequence(
      for {
        (dataType, newMapping) <- newMappings
        oldMapping <- oldMappings.get(dataType)
      } yield {
        val patchData = createPatch(newMapping, oldMapping)
        val patchIsGood = isPatchGood(patchData, newMapping.size)
        uploadPatch(dataType, patchData, patchIsGood).flatMap(_.map { transactionId =>
          storeStats(partner, dataType, matchingResults, patchData, transactionId)
        }.getOrElse(Future.successful(Nil))
        )
      }
    ).recover { case e =>
      log.error(s"failed to update partner $partner", e)
    }.map(_ => matchingResults.all)
  }

  override protected def update: Future[_] = {
    val geoSynonyms = GeoSynonyms.from(extDataService)
    val geoMatcher = new GeoMatcher(config, geobaseTree, hotelsIndex, geoSynonyms)
    db.createIfNotExists(MatchingStats.table).flatMap { _ =>
      for {
        oldMappingsAll <- GeoMappingUtils.getMappings(db, dataTypes, partners)
        newHypotheses <- Future.sequence(for {
          partner <- partners
        } yield updatePartner(partner, geoMatcher, oldMappingsAll))
      } yield storeHypotheses(newHypotheses.toIterable.flatten)
    }
  }
}