package ru.yandex.tours.indexer.google

import java.util.concurrent.atomic.AtomicInteger

import akka.actor.ActorRefFactory
import akka.stream.ActorMaterializer
import akka.stream.scaladsl.Source
import ru.yandex.tours.db.ExternalHotelInfo.ExternalInfo
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.{DBWrapper, ExternalHotelInfo}
import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import ru.yandex.tours.util.Logging
import ru.yandex.tours.util.akka.Streams
import ru.yandex.tours.util.collections.RafBasedMap
import ru.yandex.tours.util.google.GooglePlaceClient
import ru.yandex.tours.util.google.GooglePlaceClient.KeysExhausted
import slick.driver.MySQLDriver.api._

import scala.concurrent.{ExecutionContext, Future}

class GooglePlacesDownloader(db: DBWrapper, hotelsDao: HotelsDao, client: GooglePlaceClient)
                            (implicit ec: ExecutionContext, actorRefFactory: ActorRefFactory) extends Logging {
  private implicit val materializer = ActorMaterializer()
  private val parallelism = 3
  private val batchSize = 100

  db.createIfNotExists(ExternalHotelInfo.table)

  def run(): Future[Unit] = {
    hotelsDao.retrieveRafMap().flatMap { map =>
      runProcess(map).andThen {
        case _ => map.close()
      }
    }
  }

  private def getAlreadyDoneId: Future[Set[Int]] = {
    val query = for {
      ei <- ExternalHotelInfo.table
    } yield ei.id
    db.run(query.result).map(res => res.toSet)
  }

  private def runProcess(map: RafBasedMap[Int, PartnerHotel]): Future[Unit] = {
    val batchInserted = new AtomicInteger()
    getAlreadyDoneId.flatMap { usedIds =>
      Source.fromIterator(() => map.valuesIterator)
        .filter(h => !usedIds.contains(h.getId))
        .mapAsync(parallelism)(getExternalInfo)
        .via(Streams.flatten)
        .grouped(batchSize)
        .mapAsync(1) { infos =>
          val query = ExternalHotelInfo.table ++= infos
          db.run(query)
        }
        .runForeach { _ =>
          val inserted = batchInserted.incrementAndGet()
          if (inserted * batchSize % 1000 == 0) {
            log.info(s"Got ${inserted * batchSize} google responses")
          }
        }
    }
  }

  private def getExternalInfo(h: PartnerHotel): Future[Iterable[ExternalInfo]] = {
    client.search(h).map { r =>
      Iterable(ExternalInfo(h.getId, r))
    }.recover {
      case KeysExhausted => Iterable()
    }
  }
}
