package ru.yandex.tours.indexer.hotels

import java.io.File

import it.unimi.dsi.fastutil.ints.IntOpenHashSet
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import ru.yandex.tours.model.hotels.Partners
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.util.concurrent.BatchExecutor
import ru.yandex.tours.util.{Logging, ProtoIO}
import ru.yandex.tours.util.collections.RafBasedMap

import scala.collection.mutable
import scala.concurrent.{ExecutionContext, Future}

class PartnerHotelsDbUpdater(hotelsDao: HotelsDao)(implicit ec: ExecutionContext) extends Logging {
  private val BATCH_SIZE = 250

  /**
   *
   * @param oldHotels - map with old hotels
   * @param file which contains delimited new [[PartnerHotel]]
   * @return future with result of inserting hotels to db
   */
  def update(oldHotels: RafBasedMap[Int, PartnerHotel], file: File, partner: Partner): Future[Unit] = {
    val newHotels = hotels(file).filterNot(h => oldHotels.contains(h.getId))
    val toUpdate = hotels(file).filter { h =>
      oldHotels.get(h.getId) match {
        case Some(oldHotel) => oldHotel != h
        case None => false
      }
    }
    val knownHotels = hotels(file).map(_.getId).toSet
    val toDelete = for {
      hotel <- oldHotels.valuesIterator.filterNot(_.getIsDeleted)
      if !knownHotels.contains(hotel.getId)
    } yield hotel.toBuilder.setIsDeleted(true).build()

    val previous = oldHotels.count(!_._2.getIsDeleted)
    val current = hotels(file).size
    if (previous / 2 > current) {
      Future.failed(new Exception(s"Suspicious count of hotels in new index! Previous: $previous, current: $current"))
    } else {
      for {
        addCount <- add(partner, newHotels)
        updatedCount <- update(partner, toUpdate)
        deleteCount <- update(partner, toDelete)
      } yield {
        log.info(s"$partner: $addCount hotels added. $updatedCount hotels updated. $deleteCount hotels deleted.")
      }
    }
  }

  private def add(partner: Partner, hotels: Iterator[PartnerHotel]): Future[Int] = {
    BatchExecutor.executeInBatch[PartnerHotel](
      objects = hotels,
      name = s"$partner hotels inserted into database",
      batchSize = BATCH_SIZE,
      action = hotelsDao.insert
    )
  }

  private def update(partner: Partner, hotels: Iterator[PartnerHotel]): Future[Int] = {
    BatchExecutor.executeInBatch[PartnerHotel](
      objects = hotels,
      name = s"$partner hotels update in database",
      batchSize = BATCH_SIZE,
      action = hotelsDao.update
    )
  }

  private def distinct = {
    val visited = new IntOpenHashSet()
    (h: PartnerHotel) => {
      val unique = visited.add(h.getId)
      if (!unique) log.warn(s"Hotel with id ${h.getId} already seen in ${Partners(h.getRawHotel.getPartner)} feed")
      unique
    }
  }

  private def hotels(file: File) = ProtoIO.loadFromFile(file, PartnerHotel.PARSER).filter(distinct)
}
