package ru.yandex.tours.indexer.hotels

import java.io.File

import ru.yandex.tours.db.DBWrapper
import ru.yandex.tours.db.dao.HotelsDao
import ru.yandex.tours.db.dao.HotelsDao.OnlyPartner
import ru.yandex.tours.indexer.hotels.feed.PartnerFeedRetriever
import ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel
import ru.yandex.tours.model.hotels.Partners.Partner
import ru.yandex.tours.model.image.ImageProviders.ImageProvider
import ru.yandex.tours.util.collections.RafBasedMap
import ru.yandex.tours.util.lang.Futures._
import ru.yandex.tours.util.{IO, Logging}

import scala.concurrent.{ExecutionContext, Future}
import scala.util.Failure

class PartnerHotelsIndexer(db: DBWrapper,
                           hotelsDao: HotelsDao,
                           feedRetriever: PartnerFeedRetriever,
                           parser: PartnerHotelParser,
                           stableIdEnricher: StableIdsEnricher,
                           imageEnricher: ImageEnricher,
                           addInfoEnricher: AddInfoEnricher,
                           dbUpdater: PartnerHotelsDbUpdater,
                           partner: Partner,
                           imageProvider: ImageProvider)(implicit ec: ExecutionContext) extends Logging {

  private def retrieveFeed = feedRetriever.retrieve.logTiming(s"$partner feed retrieving")

  private def parseFeed(feed: File) = {
    parser.parse(feed).logTiming(s"Raw $partner hotels parsing")
  }

  private def setStableIds(hotelsFile: File): Future[File] = {
    stableIdEnricher.enrich(hotelsFile, partner).logTiming(s"Set stable ids to $partner hotels")
  }

  private def loadOldHotels(): Future[RafBasedMap[Int, PartnerHotel]] = {
    hotelsDao.retrieveRafMap(OnlyPartner(partner)).logTiming(s"Retrieving old hotels of $partner")
  }

  private def enrichImages(hotelsFile: File, oldHotels: RafBasedMap[Int, PartnerHotel]): Future[File] = {
    imageEnricher.enrich(hotelsFile, partner, imageProvider, oldHotels)
      .logTiming(s"Downloading images of $partner hotels")
  }

  private def enrichAddedInfo(oldHotels: RafBasedMap[Int, PartnerHotel], hotelsFile: File): Future[File] = {
    addInfoEnricher.enrich(hotelsFile, oldHotels, partner).logTiming(s"Set add info to $partner hotels")
  }

  private def updateHotels(oldHotels: RafBasedMap[Int, PartnerHotel], hotelFile: File): Future[Unit] = {
    dbUpdater.update(oldHotels, hotelFile, partner).logTiming(s"Update $partner hotels in database")
  }

  /**
   * Parse and store partner hotels
   * @return - file which contains parsed [[ru.yandex.tours.model.hotels.HotelsHolder.PartnerHotel]]
   */
  def index: Future[File] = {

    for {
      feed <- retrieveFeed
      rawHotels <- parseFeed(feed)
        .andThen { case _ ⇒ IO.deleteFile(feed) }
      idedHotels <- setStableIds(rawHotels)
        .andThen { case _ ⇒ IO.deleteFile(rawHotels) }
      oldHotels <- loadOldHotels()
      imaged <- enrichImages(idedHotels, oldHotels)
        .andThen { case Failure(_) ⇒ oldHotels.close() }
        .andThen { case _ ⇒ IO.deleteFile(idedHotels) }
      addInfo <- enrichAddedInfo(oldHotels, imaged)
        .andThen { case Failure(_) ⇒ oldHotels.close() }
        .andThen { case _ ⇒ IO.deleteFile(imaged) }
      _ <- updateHotels(oldHotels, addInfo)
        .andThen { case Failure(_) ⇒ IO.deleteFile(addInfo) }
        .andThen { case _ ⇒ oldHotels.close() }
    } yield addInfo
  }
}
